#!/bin/bash

# Copyright (c) 2020 Open Mobile Platform LLC.
#
# License: Proprietary

SELF_DIR=$(dirname $(readlink -f "$0"))

map_import() {
	local import="$1"
	grep "^$import " $SELF_DIR/pythonappdeps.txt|awk '{print $2}'
}

parse_imports() {
    sed -n -e '
    # Extract each "module" from lines matching:
    # "import" module ["as" name] ( "," module ["as" name] )* [";" other_statement] ["#" comment]
    /^\s*import\s\+/ {
        s///
        s/[;#].*$//
        s/\s*$//
        s/\s\+as\+[^,]\+//g
        s/,/\n/g
        s/\s\+/\n/g
        p
    }
    # Extract the "module" from lines matching:
    # "from" module "import" rest
    /^\s*from\s\+\([^[:space:]]\+\)\s\+import\s.*/ {
        s//\1/
        p
    }
    ' |sort -u
}

map_imports() {
    local imports=($(parse_imports < $1))
    for import in "${imports[@]}"
    do
        map_import "$import"
    done
}

################################################################################
if [[ $1 != --self-test ]]; then  ###  M A I N  EXECUTION BEGINS HERE  #########
################################################################################

case $1 in
-R|--requires)
	while read filename
	do
		map_imports "$filename"
	done
    ;;
--self-test)
    self_test
    ;;
*)
    cat > /dev/null
    ;;
esac

##############################################################################
exit 0; fi ###  S E L F - T E S T  EXECUTION BEGINS HERE #####################
##############################################################################

set -o nounset

temp=$(mktemp -d) || exit
trap "rm -rf '$temp'" EXIT
cd "$temp" || exit

tc_failed_num=0

compare_strings() {
    test_result="$1"
    expected="$2"
    if [[ "$test_result" = "$expected" ]]; then
        return 0
    else
        echo " Expected:"
        echo "$expected"
        echo "but got instead:"
        echo "$test_result"
        return 1
    fi
}

#test the whole chain - depends on having mappings in place for curses and sqlite3
simple_import_tc() {
    cat > test.py <<EOF
import sqlite3
import curses
import notincluded
not even import
EOF
    stdout=$(map_imports test.py)
    expected='python3dist(curses)
python3dist(sqlite3)'
    compare_strings "$stdout" "$expected"
}

import_with_extra_spaces_tc() {
    SPACE=' '
    input="import  pkg1
import pkg2$SPACE
import  pkg3"
    imports=$(parse_imports <<<"$input")
    expected='pkg1
pkg2
pkg3'
    compare_strings "$imports" "$expected"

}

import_with_comma_tc() {
    input='import pkg1,pkg2'
    imports=$(parse_imports <<<"$input")
    expected='pkg1
pkg2'
    compare_strings "$imports" "$expected"
}

from_tc() {
    input='from pkg import nopkg'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

from_with_comma_tc() {
    input='from pkg import nopkg1,nopkg2'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

import_with_comma_and_space_after_tc() {
    input='import pkg1, pkg2'
    imports=$(parse_imports <<<"$input")
    expected='pkg1
pkg2'
    compare_strings "$imports" "$expected"
}

import_with_comma_and_spaces_before_and_after_tc() {
    input='import pkg1 , pkg2'
    imports=$(parse_imports <<<"$input")
    expected='pkg1
pkg2'
    compare_strings "$imports" "$expected"
}

import_as_tc() {
    input='import pkg as nopkg'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

from_as_tc() {
    input='from pkg import nopkg1 as nopkg2'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

import_as_2_tc() {
    input='import pkg1 as nopkg1, pkg2 as nopkg2'
    imports=$(parse_imports <<<"$input")
    expected='pkg1
pkg2'
    compare_strings "$imports" "$expected"
}

from_as_2_tc() {
    input='from pkg import nopkg1 as nopkg2, nopkg3 as nopkg4'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

import_with_semicolon_tc() {
    input='import pkg; nopkg'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

from_with_semicolon_tc() {
    input='from pkg import nopkg1; nopkg2'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

import_with_comment_tc() {
    input='import pkg # nopkg'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

from_with_comment_tc() {
    input='from pkg import nopkg1 # nopkg2'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

import_in_try_tc() {
    input='try:
    import pkg
except ImportError:
    nopkg'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

from_in_try_tc() {
    input='try:
    from pkg import nopkg
except ImportError:
    nopkg'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

conditional_import_tc() {
    input='if USE_FOO:
    import pkg'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

conditional_from_tc() {
    input='if USE_FOO:
    from pkg import nopkg'
    imports=$(parse_imports <<<"$input")
    expected='pkg'
    compare_strings "$imports" "$expected"
}

run_tc() {
    local tc=$1
    echo -n "*** Executing test case: $tc:"
    ${tc}_tc
    local rc=$?
    if [[ $rc -ne 0 ]]; then
        let tc_failed_num++;
        echo -e " \e[31mFAILED\e[0m!"
    else
        echo " PASSED"
    fi
}

run_tc simple_import
run_tc import_with_extra_spaces
run_tc import_with_comma
run_tc from
run_tc from_with_comma
run_tc import_with_comma_and_space_after
run_tc import_with_comma_and_spaces_before_and_after
run_tc import_as
run_tc from_as
run_tc import_as_2
run_tc from_as_2
run_tc import_with_semicolon
run_tc from_with_semicolon
run_tc import_with_comment
run_tc from_with_comment
run_tc import_in_try
run_tc from_in_try
run_tc conditional_import
run_tc conditional_from

echo "$tc_failed_num tests failed"