/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Gallery 1.0
import Sailfish.Gallery.private 1.0

ZoomableFlickable {
    id: flickable

    property string mimeType
    property bool active: true
    property alias source: photo.source
    property alias photo: photo
    property alias largePhoto: largePhoto
    readonly property bool _active: active || viewMoving
    readonly property bool error: photo.status == Image.Error
    readonly property bool isSvg: mimeType === "image/svg+xml"
    readonly property real zoomStep: 1.2

    signal clicked
    signal doubleClicked

    function setLargePhoto() {
        if (largePhoto.source != photo.source) {
            largePhoto.source = photo.source
        }
    }

    implicitContentWidth: photo.implicitWidth
    implicitContentHeight: photo.implicitHeight
    contentRotation: -metadata.orientation
    scrollDecoratorColor: Theme.lightPrimaryColor
    zoomEnabled: photo.status == Image.Ready
    maximumZoom: Math.max(Screen.width, Screen.height) / 200
                 * Math.max(1.0, photo.implicitWidth > 0 ? largePhoto.implicitHeight / photo.implicitHeight
                                                         : 1.0)

    on_ActiveChanged: {
        if (!_active) {
            resetZoom()

            if (isSvg) {
                setLargePhoto()
            } else {
                largePhoto.source = ""
            }
        }
    }

    onIsSvgChanged: if (isSvg) setLargePhoto()
    onAboutToZoom: setLargePhoto()

    Image {
        id: photo
        objectName: "zoomableImage"

        property var errorLabel

        anchors.fill: parent
        smooth: !(movingVertically || movingHorizontally)
        fillMode: Image.PreserveAspectFit
        visible: largePhoto.status !== Image.Ready
        asynchronous: true
        cache: false
        opacity: status == Image.Ready ? 1 : 0

        onStatusChanged: {
            if (status == Image.Error) {
                errorLabel = errorLabelComponent.createObject(flickable)
            }
        }

        onSourceChanged: {
            if (errorLabel) {
                errorLabel.destroy()
                errorLabel = null
            }

            resetZoom()
        }

        sourceSize {
            width: Math.min(Screen.width, metadata.width)
            height: Math.min(Screen.height, metadata.height)
        }

        Behavior on opacity { FadeAnimation {} }
    }

    AnimatedImage {
        id: largePhoto

        anchors.fill: parent
        cache: false
        asynchronous: true
    }

    Item {
        anchors.centerIn: parent
        width: flickable.transpose ? parent.height : parent.width
        height: flickable.transpose ? parent.width : parent.height
        rotation: -flickable.contentRotation

        MouseArea {
            x: width > parent.width
               ? (parent.width - width) / 2
               : flickable.contentX + Theme.paddingLarge
            y: height > parent.height
               ? (parent.height - height) / 2
               : flickable.contentY + Theme.paddingLarge
            width: flickable.width - (2 * Theme.paddingLarge)
            height: flickable.height - (2 * Theme.paddingLarge)

            onClicked: flickable.clicked()
            onDoubleClicked: flickable.doubleClicked()
            onWheel: {
                if (wheel.modifiers && Qt.ControlModifier) {
                    const zoomFactor = wheel.angleDelta.y > 0 ? zoomStep : 1.0 / zoomStep;
                    const localPos = mapToItem(flickable.contentItem, wheel.x, wheel.y)

                    flickable.zoomIn(
                        zoomFactor,
                        localPos,
                        localPos
                    )
                }
            }
        }
    }

    ImageMetadata {
        id: metadata

        source: photo.source
        autoUpdate: false
    }

    BusyIndicator {
        anchors.centerIn: parent
        parent: flickable
        running: photo.status === Image.Loading && !delayBusyIndicator.running
        size: BusyIndicatorSize.Large

        Timer {
            id: delayBusyIndicator

            running: photo.status === Image.Loading
            interval: 1000
        }
    }

    Component {
        id: errorLabelComponent

        ErrorItem { }
    }
}
