/****************************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2015 Inigo Quilez
** SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: Proprietary
**
** This file contains a modified version of "sdRoundBox function".
** "sdRoundBox function" was released by Inigo Quilez under MIT License and available
** at https://www.shadertoy.com/view/4llXD7.
** License text is available at https://www.shadertoy.com/view/4llXD7.
**
****************************************************************************************/

import QtQuick 2.5
import QtQuick.Window 2.2 as Q

import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Sailfish.Silica.Background 1.0

import Aurora.Controls.private 1.0

SilicaItem {
    id: root

    property real radius: Theme.paddingSmall
    property alias color: shader.color
    property int roundedCorners: Corners.All

    implicitWidth: Theme.dp(350)
    implicitHeight: Theme.itemSizeMedium

    ShaderEffect {
        id: shader

        property real progress: SkeletonAnimation.progress
        property real progress2: SkeletonAnimation.progress2
        property real gradWidth: SkeletonAnimation.gradWidth
        property color color: palette.highlightColor
        property size viewSize: Qt.size(width * 0.5, height * 0.5)
        property size fbSize: Qt.size(Q.Window.width, Q.Window.height)

        property vector4d waveDirection: {
            const angle = Q.Screen.angleBetween(__silica_applicationwindow_instance.pageStack._currentOrientation,
                                                      Q.Screen.primaryOrientation)
            switch (angle) {
            case 0: return Qt.vector4d(1, 0, 0, 0)
            case 90: return Qt.vector4d(0, -1, 0, 1)
            case 180: return Qt.vector4d(-1, 0, 1, 0)
            case 270: return Qt.vector4d(0, 1, 0, 0)
            }
        }

        property real radius: Math.max(0, Math.min(root.radius, Math.min(width, height) * 0.5))
        property vector4d radiusVec4: Qt.vector4d(((roundedCorners & Corners.TopRight) >>> 1) * radius,
                                                  ((roundedCorners & Corners.BottomRight) >>> 2) * radius,
                                                  ((roundedCorners & Corners.TopLeft) /*>>> 0*/) * radius,
                                                  ((roundedCorners & Corners.BottomLeft) >>> 3) * radius)

        anchors.fill: parent

        vertexShader: "
            attribute highp vec4 qt_Vertex;
            attribute highp vec2 qt_MultiTexCoord0;
            uniform highp mat4 qt_Matrix;

            uniform mediump vec2 viewSize;
            varying mediump vec2 rectCoord;

            void main() {
                rectCoord = (2.0 * viewSize * qt_MultiTexCoord0) - viewSize;
                gl_Position = qt_Matrix * qt_Vertex;
            }
        "

        fragmentShader: "
            precision lowp float;

            varying mediump vec2 rectCoord;

            uniform float qt_Opacity;

            uniform mediump float   progress;
            uniform mediump float   progress2;
            uniform mediump float   gradWidth;
            uniform lowp    vec4    color;
            uniform mediump vec2    viewSize;
            uniform mediump vec2    fbSize;
            uniform lowp    vec4    waveDirection;
            uniform mediump vec4    radiusVec4;

            // b.x = width
            // b.y = height
            // r.x = roundness top-right
            // r.y = roundness boottom-right
            // r.z = roundness top-left
            // r.w = roundness bottom-left
            float sdRoundBox(vec2 p, vec2 b, vec4 r)
            {
                r.xy = (p.x > 0.0) ? r.xy : r.zw;
                r.x  = (p.y > 0.0) ? r.x  : r.y;
                vec2 q = abs(p) - b + r.x;

                return -(min(max(q.x, q.y), 0.0) + length(max(q, 0.0)) - r.x);
            }

            float wave(float progress) {
                mediump vec2  coord = (fbSize * waveDirection.zw) + (gl_FragCoord.xy * waveDirection.xy);
                mediump float wavePos = (waveDirection.x == 0.0 ? coord.y : coord.x);
                float wave = (-abs(wavePos - progress)  / gradWidth + 1.0) * 0.15;
                return wave > 0.0 ? wave : 0.0;
            }

            // The final color is composed the following way:
            // 1. take the base color and reduce it to 15% opacity;
            // 2. calculate the opacity gradient: 0.0 (0%) -- 0.5 * gradWidth (15%) -- 1.0 * gradWidth (0%);
            // 3. take the base color and apply the gradient opacity to it;
            // 4. blend 3 over 1.
            void main() {
                float rect = sdRoundBox(rectCoord, viewSize, radiusVec4);
                rect = clamp(rect, 0.0, 1.0);

                vec4 col = color * 0.15; // base fill
                float wave1 = wave(progress);
                col = mix(col, color, wave1);  // gradient overlay

                float wave2 = wave(progress2);
                col = mix(col, color, wave2);  // gradient2 overlay

                gl_FragColor = col * rect * qt_Opacity;
            }
        "
    }

    Binding {
        target: SkeletonAnimation
        property: "pageWidth"
        value: {
            var width = 0
            if (Q.Screen.primaryOrientation === Qt.PortraitOrientation) {
                __silica_applicationwindow_instance.pageStack._currentOrientation & Orientation.PortraitMask
                        ? width = shader.fbSize.width
                        : width = shader.fbSize.height
            } else {
                __silica_applicationwindow_instance.pageStack._currentOrientation & Orientation.LandscapeMask
                        ? width = shader.fbSize.width
                        : width = shader.fbSize.height
            }

            return width
        }
    }

    Component.onCompleted: SkeletonAnimation.users++
    Component.onDestruction: SkeletonAnimation.users--
}
