/*
 * SPDX-FileCopyrightText: 2018-2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef SAILFISH_MDM_PROXY_H
#define SAILFISH_MDM_PROXY_H

#include <QObject>
#include <QUrl>
#include <QScopedPointer>

namespace Sailfish {
namespace Mdm {

class NetProxyPrivate;

// Class for querying and configuring separate proxies
class Q_DECL_EXPORT NetProxy : public QObject
{
    Q_OBJECT
    Q_DECLARE_PRIVATE(NetProxy)
    Q_PROPERTY(bool valid READ isValid NOTIFY validChanged);

    // TODO tie up with the appropriate functionality
    Q_PROPERTY(ProxyType type READ type NOTIFY typeChanged WRITE setType);
    Q_PROPERTY(QList<QUrl> manualProxies READ manualProxies NOTIFY manualProxiesChanged WRITE setManualProxies);
    Q_PROPERTY(QStringList manualExclusions READ manualExclusions NOTIFY manualExclusionsChanged WRITE setManualExclusions);
    Q_PROPERTY(QUrl automaticUrl READ automaticUrl NOTIFY automaticUrlChanged WRITE setAutomaticUrl);
    Q_PROPERTY(ActiveType activeType READ activeType NOTIFY activeTypeChanged WRITE setActiveType);

public:
    enum ProxyType {
        Invalid = 0,
        Direct  = 1,
        Manual = 2,
        Automatic = 3
    };
    Q_ENUM(ProxyType)

    enum ActiveType {
        Service = 0,
        Global = 1
    };
    Q_ENUM(ActiveType)

    NetProxy(QObject *parent = nullptr);
    virtual ~NetProxy();

    bool isValid() const;

    // Find the proxies that are available
    QStringList getTechnologies() const;
    QStringList getIdentifiers(const QString &technology) const;

    // Use service identifier, "global" for global proxy, or "" to unset
    // By default the global proxy is selected
    bool selectProxy(const QString &identifier);
    QString getIdentifier() const;
    QString getName() const;

    // Whether service or global proxy should be applied
    bool setActiveType(ActiveType type);
    ActiveType activeType() const;

    // Configure currently selected proxy
    bool setType(ProxyType type);
    bool setManualProxies(const QList<QUrl> &proxies);
    bool setManualExclusions(const QStringList &exclusions);
    bool setAutomaticUrl(const QUrl &url);

    // Query currently selected proxy
    ProxyType type() const;
    QList<QUrl> manualProxies() const;
    QStringList manualExclusions() const;
    QUrl automaticUrl() const;

signals:
    // Triggered when the object is ready for use
    void validChanged(bool valid);

    // Triggered if the active type switches between service and global
    void activeTypeChanged(ActiveType type);

    // Triggered if the currently selected proxy configuration changes
    void proxyConfigChanged();

    // Triggered by changes to specific properties of the selected proxy
    // configuration
    void typeChanged(ProxyType type);
    void manualProxiesChanged(const QList<QUrl> &proxies);
    void manualExclusionsChanged(const QStringList &exclusions);
    void automaticUrlChanged(const QUrl &url);

private:
    QScopedPointer<NetProxyPrivate> const d_ptr;
};

} // namespace Mdm
} // namespace Sailfish

#endif // SAILFISH_MDM_PROXY_H
