/**
 * SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.Thumbnailer 1.0
import QtGraphicalEffects 1.0

Item {
    id: root

    property bool needToDisplayIcon: true
    property bool isDir: false
    property bool isArchive: false
    property string path: ""
    property string mimeType: ""
    property int iconSize: Theme.iconSizeMedium
    property int backgroundRadius
    property int backgroundWidth
    property int backgroundHeight

    readonly property var iconSizeToStringMap: {
        var iconSizeMap = ({})
        iconSizeMap[Theme.iconSizeExtraLarge] = "xl"
        iconSizeMap[Theme.iconSizeLarge] = "l"
        iconSizeMap[Theme.iconSizeMedium] = "m"
        iconSizeMap[Theme.iconSizeSmallPlus] = "splus"
        iconSizeMap[Theme.iconSizeSmall] = "s"
        return iconSizeMap
    }

    function iconForMimeType() {
        var iconPath
        if (iconSizeToStringMap[root.iconSize] == null) {
            console.warn("Incorrect choosen icon's size")
            return
        }

        if (root.isDir) {
            iconPath = "image://theme/icon-m-file-folder"
        } else {
            if (root.isArchive) {
                iconPath = "image://theme/icon-m-file-archive-folder"
            } else {
                iconPath = Theme.iconForMimeType(root.mimeType)
            }
        }

        return iconPath.replace("-m-", "-%1-".arg(iconSizeToStringMap[root.iconSize]))
    }

    width: backgroundWidth
    height: previewLoader.height

    Loader {
        id: previewLoader

        width: height
        height: sourceComponent == null ? 0 : sourceComponent.height
        anchors.horizontalCenter: parent.horizontalCenter
        sourceComponent: FileManager.isPicture(root.mimeType) || FileManager.isVideo(root.mimeType)
                         ? previewPictureComponent
                         : root.needToDisplayIcon
                            ? iconComponent
                            : null
    }

    Component {
        id: iconComponent

        Rectangle {
            id: background

            anchors.horizontalCenter: parent.horizontalCenter

            width: root.backgroundWidth
            height: root.backgroundHeight
            radius: root.backgroundRadius

            layer.enabled: true
            layer.effect: OpacityMask {
                source: background
                maskSource: Rectangle {
                    width: background.width
                    height: background.height
                    radius: background.radius
                }
            }

            gradient: Gradient {
                GradientStop { position: 0.0; color: Theme.rgba(Theme.primaryColor, 0.15) }
                GradientStop { position: 1.0; color: "transparent" }
            }

            HighlightImage {
                objectName: "IconHighlightImage"

                anchors.centerIn: parent
                cache: true
                smooth: false
                source: iconForMimeType()
            }
        }
    }

    Component {
        id: previewPictureComponent

        Thumbnail {
            id: previewPicture

            readonly property int maxHeight: Theme.itemSizeHuge + Theme.itemSizeSmall

            objectName: "PreviewPictureThumbnail"

            anchors.horizontalCenter: parent.horizontalCenter
            width: parent.width - 2 * Theme.horizontalPageMargin
            height: maxHeight

            sourceSize {
                width: previewPicture.width
                height: previewPicture.height
            }

            mimeType: root.mimeType
            priority: Thumbnail.HighPriority
            fillMode: Thumbnail.PreserveAspectFit
            source: root.path.length > 0 ? root.path : ""

            BusyIndicator {
                id: thumbnailBusyIndicator

                objectName: "ThumbnailBusyIndicator"

                anchors.centerIn: parent
                z: parent.z + 1
                running: previewPicture.status === Thumbnail.Loading
                visible: running
                size: BusyIndicatorSize.Medium

                FadeAnimator {
                    target: previewPicture
                    easing.type: Easing.InExpo
                    to: 1.0
                    running: !thumbnailBusyIndicator.running
                }
            }

            // For videos that don't have a preview
            Rectangle {
                anchors.horizontalCenter: parent.horizontalCenter
                width: Theme.itemSizeExtraLarge * 4 - 2 * Theme.horizontalPageMargin
                height: previewPicture.maxHeight
                visible: previewPicture.status === Thumbnail.Error
                color: "black"
            }
        }
    }
}
