/**
 * SPDX-FileCopyrightText: 2017-2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2021-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef SAILFISH_MDM_DEVICELOCK_H
#define SAILFISH_MDM_DEVICELOCK_H

#include <QObject>
#include <QVariantMap>
#include <QDBusError>
#include <QDBusConnection>
#include <QDBusInterface>
#include <QDBusServiceWatcher>

namespace Sailfish {
namespace Mdm {

class DeviceLockPrivate;
class Q_DECL_EXPORT DeviceLock : public QObject
{
    Q_OBJECT
    Q_PROPERTY(State state READ state NOTIFY stateChanged)
    Q_PROPERTY(bool settingLockoutMode READ isSettingLockoutMode NOTIFY settingLockoutModeChanged)
    Q_PROPERTY(bool clearingDevice READ isClearingDevice NOTIFY clearingDeviceChanged)
    Q_PROPERTY(bool changingSecurityCode READ isChangingSecurityCode NOTIFY changingSecurityCodeChanged)
    Q_PROPERTY(bool homeEncrypted READ isHomeEncrypted)
public:
    enum State {
        Unlocked,
        Locked,
        ManagerLockout,
        CodeEntryLockout,
        Undefined
    };
    Q_ENUM(State)

    enum Result {
        Success                 =  0,
        Failure                 = -1,
        SecurityCodeExpired     = -2,
        SecurityCodeInHistory   = -3,
        LockedOut               = -4,
        Evaluating              = -5,
        UserSettingsSyntaxFail  = -6,
        LoginTTLFail            = -7,
        LoginScheduleFail       = -8,
        LoginPassFail           = -9,
        NoMeetRequirements      = -10,
        NoSuchUser              = -11,
        VerifyTokenError        = -12
    };
    Q_ENUM(Result)

    enum ClearOption {
        ShutdownAfterClear  = 0x00,
        RebootAfterClear    = 0x01,
        WipePartitions      = 0x02,
        ClearExternalMedia  = 0x04
    };
    Q_ENUM(ClearOption)
    Q_DECLARE_FLAGS(ClearOptions, ClearOption)
    Q_FLAG(ClearOptions)

    enum ManagerLockoutMode {
        NoManagerLockout,
        RecoverableManagerLockout,
        PermanentManagerLockout
    };

    explicit DeviceLock(QObject *parent = nullptr);
    ~DeviceLock();

    State state() const;

    Q_DECL_DEPRECATED static DeviceLock::Result authInputResult(const QDBusError &err);
    Q_DECL_DEPRECATED static DeviceLock::Result authInputResult(const QString &err);

    ClearOptions supportedResetOptions() const;

    bool isSettingLockoutMode() const;
    Q_INVOKABLE void setLockoutMode(ManagerLockoutMode mode, const QVariantMap &feedback = QVariantMap());
    Q_INVOKABLE void setLockoutWarning(const QString &warning);

    bool isClearingDevice() const;
    Q_INVOKABLE void clearDevice(ClearOptions options = ShutdownAfterClear);

    bool isChangingSecurityCode() const;
    Q_INVOKABLE void setSecurityCode(const QString &code);

    Q_INVOKABLE bool isHomeEncrypted();

    void expirePassword(uid_t uid);
    void expirePasswordForAll();

    void setOneTimePassword(uid_t uid, const QString &password);
    void clearPassword(uid_t uid);
    void setSecurityCodeUid(uid_t uid, const QString &code);
    Q_DECL_DEPRECATED void updateSupportedOptions();

public slots:
    Q_DECL_DEPRECATED void handleUserMethodActionCompleted(uint uid,
                                                           const QString &method,
                                                           const QString &action);
    Q_DECL_DEPRECATED void handleMethodActionCompleted(const QString &method,
                                                       const QString &action);
    Q_DECL_DEPRECATED void handleMethodOptionChanged(const QString &method,
                                                     const QString &key,
                                                     const QDBusVariant &value);
    Q_DECL_DEPRECATED void handleStateChanged(int state);
signals:
    void stateChanged();

    void lockoutModeChanged();
    void lockoutModeError();
    void settingLockoutModeChanged();
    void settingLockoutWarningFinished(DeviceLock::Result err);

    void deviceCleared();
    void clearDeviceError();
    void clearingDeviceChanged();

    void securityCodeChanged();
    void securityCodeChangeError(DeviceLock::Result err);
    void changingSecurityCodeChanged();

    void expirePasswordChanged();
    void expirePasswordError(DeviceLock::Result err);

    void expirePasswordForAllChanged();
    void expirePasswordForAllError();
    void setOneTimePasswordChanged();
    void setOneTimePasswordError(DeviceLock::Result err);

    void clearPasswordChanged();
    void clearPasswordError(Result err);

    void securityCodeUidChanged();
    void securityCodeUidChangeError(DeviceLock::Result err);

private:
    DeviceLockPrivate *d_ptr;
    Q_DISABLE_COPY(DeviceLock)
    Q_DECLARE_PRIVATE(DeviceLock)
};

}
}

Q_DECLARE_OPERATORS_FOR_FLAGS(Sailfish::Mdm::DeviceLock::ClearOptions)

#endif
