/***************************************************************************
**
** SPDX-FileCopyrightText: 2020-2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: LGPL-2.1-only
**
****************************************************************************/

#ifndef MPERMISSION_H_
#define MPERMISSION_H_

#include <mlite-global.h>

class MDesktopEntry;
class MPermissionPrivate;

/*!
 * MPermission is subject to change in future versions.
 *
 * \internal
 */

class MLITESHARED_EXPORT MPermission
{
public:
    /*!
     * Reads permission file for the permission and constructs a new
     * MPermission instance.
     *
     * \param file name of the permission file to read.
     */
    MPermission(const QString &fileName);

    /*!
     * Copy constructor.
     *
     * \param instance to copy.
     */
    MPermission(const MPermission &other);

    /*!
     * Destroys MPermission instance.
     */
    ~MPermission();

    /*!
     * Constructs a list of MPermission instances from permissions named
     * by MDesktopEntry.
     *
     * \param MDesktopEntry to read permissions from.
     */
    static QList<MPermission> fromDesktopEntry(const MDesktopEntry &entry);

    /*!
     * Returns name for the permission.
     */
    QString name() const;

    /*!
     * Returns whether the permission is valid and contains the needed
     * information.
     */
    bool isValid() const;

    /*!
     * Returns description for the permission. Localized if available,
     * otherwise the same as descriptionUnlocalized.
     */
    QString description() const;

    /*!
     * Returns unlocalized description for the permission. Empty string
     * if the permission is not valid.
     */
    QString descriptionUnlocalized() const;

    /*!
     * Returns long description for the permission that complements
     * description and provides more information about the permission.
     * Localized if available, otherwise the same as
     * longDescriptionUnlocalized.
     */
    QString longDescription() const;

    /*!
     * Returns unlocalized long description for the permission. Empty
     * string if the permission is not valid.
     */
    QString longDescriptionUnlocalized() const;

    /*!
     * Return icon name from theme (if it present in permission file, or empty string)
     */
    QString icon() const;

    MPermission &operator=(const MPermission &);
protected:
    /*! \internal */
    //! Pointer to the private class
    MPermissionPrivate *const d_ptr;
    /*! \internal_end */

private:
    Q_DECLARE_PRIVATE(MPermission);
};

#endif /* MPERMISSION_H_ */
