/*
 * Copyright (c) 2017 - 2020 Jolla Ltd.
 * Copyright (c) 2019 - 2020 Open Mobile Platform LLC.
 *
 * License: Proprietary
 */

#ifndef SAILFISH_MDM_ACCOUNTS_H
#define SAILFISH_MDM_ACCOUNTS_H

#include <QObject>
#include <QDateTime>
#include <QVariantMap>
#include <QString>
#include <QList>

#include <QDBusArgument>
#include <QDBusMetaType>

namespace Sailfish {
namespace Mdm {
namespace Accounts {

class ProvisionerPrivate;
class Q_DECL_EXPORT Provisioner : public QObject
{
    Q_OBJECT
    Q_PROPERTY(bool canCreateAccount READ canCreateAccount NOTIFY canCreateAccountChanged)
    Q_PROPERTY(QString providerName READ providerName WRITE setProviderName NOTIFY providerNameChanged)
    Q_PROPERTY(QString accountDisplayName READ accountDisplayName WRITE setAccountDisplayName NOTIFY accountDisplayNameChanged)
    Q_PROPERTY(QVariantMap accountConfiguration READ accountConfiguration WRITE setAccountConfiguration NOTIFY accountConfigurationChanged)
    Q_PROPERTY(bool accountReadOnly READ accountReadOnly WRITE setAccountReadOnly NOTIFY accountReadOnlyChanged)
    Q_PROPERTY(bool accountLimited READ accountLimited WRITE setAccountLimited NOTIFY accountLimitedChanged)
    Q_PROPERTY(int user READ user WRITE setUser NOTIFY userChanged)
    Q_PROPERTY(QString lastTransactionUuid READ lastTransactionUuid NOTIFY lastTransactionUuidChanged)
    Q_PROPERTY(bool queuedTransactionUsed READ queuedTransactionUsed NOTIFY queuedTransactionUsedChanged)

public:
    Provisioner(QObject *parent = nullptr);
    virtual ~Provisioner();

    QString providerName() const;
    void setProviderName(const QString &providername);
    QString accountDisplayName() const;
    void setAccountDisplayName(const QString &displayName);
    QVariantMap accountConfiguration() const;
    void setAccountConfiguration(const QVariantMap &config);

    bool accountReadOnly() const;
    void setAccountReadOnly(bool readOnly);

    bool accountLimited() const;
    void setAccountLimited(bool limited);

    void setResultHandler(const QString &service, const QString &path,
                          const QString &interface, const QString &method);

    int user() const;
    void setUser(int user);

    QString lastTransactionUuid() const;
    bool queuedTransactionUsed() const;

    int error() const;
    QString errorMessage() const;

    struct AccountInfo {
        AccountInfo() : accountId(0) {}
        AccountInfo(const AccountInfo &other)
            : accountId(other.accountId)
            , providerName(other.providerName)
            , remoteIdentifier(other.remoteIdentifier) {}
        quint64 accountId;
        QString providerName;
        QString remoteIdentifier;
    };
    bool canCreateAccount() const;
    virtual bool createAccount() = 0;
    virtual bool deleteAccount(quint64 accountId);
    virtual QList<Sailfish::Mdm::Accounts::Provisioner::AccountInfo> listAccounts();
    virtual bool selectAccount(quint64 accountId);
    virtual bool commitChanges();

Q_SIGNALS:
    void canCreateAccountChanged();
    void providerNameChanged();
    void accountDisplayNameChanged();
    void accountConfigurationChanged();
    void accountReadOnlyChanged();
    void accountLimitedChanged();
    void userChanged();
    void lastTransactionUuidChanged();
    void queuedTransactionUsedChanged();

protected:
    void setPrivatePtr(Sailfish::Mdm::Accounts::ProvisionerPrivate *dptr);

private:
    Sailfish::Mdm::Accounts::ProvisionerPrivate *d_ptr;
    Q_DISABLE_COPY(Provisioner)
    Q_DECLARE_PRIVATE(Provisioner)
};

class GenericAccountProvisionerPrivate;
class Q_DECL_EXPORT GenericAccountProvisioner : public Provisioner
{
    Q_OBJECT
    Q_PROPERTY(QString defaultServiceName READ defaultServiceName WRITE setDefaultServiceName NOTIFY defaultServiceNameChanged)
    Q_PROPERTY(QString username READ username WRITE setUsername NOTIFY usernameChanged)
    Q_PROPERTY(QString password READ password WRITE setPassword NOTIFY passwordChanged)

public:
    GenericAccountProvisioner(QObject *parent = nullptr);

    QString defaultServiceName() const;
    void setDefaultServiceName(const QString &servicename);
    QString username() const;
    void setUsername(const QString &username);
    QString password() const;
    void setPassword(const QString &password);

    bool createAccount() Q_DECL_OVERRIDE;

Q_SIGNALS:
    void defaultServiceNameChanged();
    void usernameChanged();
    void passwordChanged();

private:
    Sailfish::Mdm::Accounts::GenericAccountProvisionerPrivate *d_ptr;
    Q_DISABLE_COPY(GenericAccountProvisioner)
    Q_DECLARE_PRIVATE(GenericAccountProvisioner)
};

class EmailAccountProvisionerPrivate;
class Q_DECL_EXPORT EmailAccountProvisioner : public Provisioner
{
    Q_OBJECT
    Q_PROPERTY(QString defaultServiceName READ defaultServiceName WRITE setDefaultServiceName NOTIFY defaultServiceNameChanged)
    Q_PROPERTY(QString emailAddress READ emailAddress WRITE setEmailAddress NOTIFY emailAddressChanged)
    Q_PROPERTY(Sailfish::Mdm::Accounts::EmailAccountProvisioner::IncomingProtocol incomingProtocol READ incomingProtocol WRITE setIncomingProtocol NOTIFY incomingProtocolChanged)
    Q_PROPERTY(QString incomingUsername READ incomingUsername WRITE setIncomingUsername NOTIFY incomingUsernameChanged)
    Q_PROPERTY(QString incomingPassword READ incomingPassword WRITE setIncomingPassword NOTIFY incomingPasswordChanged)
    Q_PROPERTY(QString incomingServer READ incomingServer WRITE setIncomingServer NOTIFY incomingServerChanged)
    Q_PROPERTY(int incomingPort READ incomingPort WRITE setIncomingPort NOTIFY incomingPortChanged)
    Q_PROPERTY(Sailfish::Mdm::Accounts::EmailAccountProvisioner::ConnectionEncryption incomingConnectionEncryption READ incomingConnectionEncryption WRITE setIncomingConnectionEncryption NOTIFY incomingConnectionEncryptionChanged)
    Q_PROPERTY(bool incomingAcceptUntrustedCertificates READ incomingAcceptUntrustedCertificates WRITE setIncomingAcceptUntrustedCertificates NOTIFY incomingAcceptUntrustedCertificatesChanged)
    Q_PROPERTY(Sailfish::Mdm::Accounts::EmailAccountProvisioner::OutgoingProtocol outgoingProtocol READ outgoingProtocol WRITE setOutgoingProtocol NOTIFY outgoingProtocolChanged)
    Q_PROPERTY(QString outgoingUsername READ outgoingUsername WRITE setOutgoingUsername NOTIFY outgoingUsernameChanged)
    Q_PROPERTY(QString outgoingPassword READ outgoingPassword WRITE setOutgoingPassword NOTIFY outgoingPasswordChanged)
    Q_PROPERTY(QString outgoingServer READ outgoingServer WRITE setOutgoingServer NOTIFY outgoingServerChanged)
    Q_PROPERTY(int outgoingPort READ outgoingPort WRITE setOutgoingPort NOTIFY outgoingPortChanged)
    Q_PROPERTY(Sailfish::Mdm::Accounts::EmailAccountProvisioner::ConnectionEncryption outgoingConnectionEncryption READ outgoingConnectionEncryption WRITE setOutgoingConnectionEncryption NOTIFY outgoingConnectionEncryptionChanged)
    Q_PROPERTY(bool outgoingAcceptUntrustedCertificates READ outgoingAcceptUntrustedCertificates WRITE setOutgoingAcceptUntrustedCertificates NOTIFY outgoingAcceptUntrustedCertificatesChanged)
    Q_PROPERTY(Sailfish::Mdm::Accounts::EmailAccountProvisioner::Authentication outgoingAuthentication READ outgoingAuthentication WRITE setOutgoingAuthentication NOTIFY outgoingAuthenticationChanged)

public:
    enum IncomingProtocol {
        NoIncomingProtocol  = 0,
        Imap4               = 1,
        Pop3                = 2
    };
    Q_ENUM(IncomingProtocol)

    enum OutgoingProtocol {
        NoOutgoingProtocol  = 0,
        Smtp                = 1
    };
    Q_ENUM(OutgoingProtocol)

    enum ConnectionEncryption {
        NoEncryption        = 0,
        SSL                 = 1,
        StartTLS            = 2
    };
    Q_ENUM(ConnectionEncryption)

    enum Authentication {
        NoAuthentication    = 0,
        Login               = 1,
        Plain               = 2,
        CramMd5             = 3
    };
    Q_ENUM(Authentication)

    EmailAccountProvisioner(QObject *parent = nullptr);

    QString defaultServiceName() const;
    void setDefaultServiceName(const QString &servicename);
    QString emailAddress() const;
    void setEmailAddress(const QString &emailaddress);

    Sailfish::Mdm::Accounts::EmailAccountProvisioner::IncomingProtocol incomingProtocol() const;
    void setIncomingProtocol(Sailfish::Mdm::Accounts::EmailAccountProvisioner::IncomingProtocol protocol);
    QString incomingUsername() const;
    void setIncomingUsername(const QString &username);
    QString incomingPassword() const;
    void setIncomingPassword(const QString &password);
    QString incomingServer() const;
    void setIncomingServer(const QString &server);
    int incomingPort() const;
    void setIncomingPort(int port);
    Sailfish::Mdm::Accounts::EmailAccountProvisioner::ConnectionEncryption incomingConnectionEncryption() const;
    void setIncomingConnectionEncryption(Sailfish::Mdm::Accounts::EmailAccountProvisioner::ConnectionEncryption enc);
    bool incomingAcceptUntrustedCertificates() const;
    void setIncomingAcceptUntrustedCertificates(bool accept);

    Sailfish::Mdm::Accounts::EmailAccountProvisioner::OutgoingProtocol outgoingProtocol() const;
    void setOutgoingProtocol(Sailfish::Mdm::Accounts::EmailAccountProvisioner::OutgoingProtocol protocol);
    QString outgoingUsername() const;
    void setOutgoingUsername(const QString &username);
    QString outgoingPassword() const;
    void setOutgoingPassword(const QString &password);
    QString outgoingServer() const;
    void setOutgoingServer(const QString &server);
    int outgoingPort() const;
    void setOutgoingPort(int port);
    Sailfish::Mdm::Accounts::EmailAccountProvisioner::ConnectionEncryption outgoingConnectionEncryption() const;
    void setOutgoingConnectionEncryption(Sailfish::Mdm::Accounts::EmailAccountProvisioner::ConnectionEncryption enc);
    bool outgoingAcceptUntrustedCertificates() const;
    void setOutgoingAcceptUntrustedCertificates(bool accept);
    Sailfish::Mdm::Accounts::EmailAccountProvisioner::Authentication outgoingAuthentication() const;
    void setOutgoingAuthentication(Sailfish::Mdm::Accounts::EmailAccountProvisioner::Authentication requiresAuth);

    bool createAccount() Q_DECL_OVERRIDE;
    bool selectAccount(quint64 accountId) Q_DECL_OVERRIDE;
    bool commitChanges() Q_DECL_OVERRIDE;

Q_SIGNALS:
    void defaultServiceNameChanged();
    void emailAddressChanged();
    void incomingProtocolChanged();
    void incomingUsernameChanged();
    void incomingPasswordChanged();
    void incomingServerChanged();
    void incomingPortChanged();
    void incomingConnectionEncryptionChanged();
    void incomingAcceptUntrustedCertificatesChanged();
    void outgoingProtocolChanged();
    void outgoingUsernameChanged();
    void outgoingPasswordChanged();
    void outgoingServerChanged();
    void outgoingPortChanged();
    void outgoingConnectionEncryptionChanged();
    void outgoingAcceptUntrustedCertificatesChanged();
    void outgoingAuthenticationChanged();

private:
    Sailfish::Mdm::Accounts::EmailAccountProvisionerPrivate *d_ptr;
    Q_DISABLE_COPY(EmailAccountProvisioner)
    Q_DECLARE_PRIVATE(EmailAccountProvisioner)
};

class ActiveSyncAccountProvisionerPrivate;
class Q_DECL_EXPORT ActiveSyncAccountProvisioner : public Provisioner
{
    Q_OBJECT
    Q_PROPERTY(QString username READ username WRITE setUsername NOTIFY usernameChanged)
    Q_PROPERTY(QString password READ password WRITE setPassword NOTIFY passwordChanged)
    Q_PROPERTY(QString emailAddress READ emailAddress WRITE setEmailAddress NOTIFY emailAddressChanged)
    Q_PROPERTY(QString description READ description WRITE setDescription NOTIFY descriptionChanged)
    Q_PROPERTY(QString signature READ signature WRITE setSignature NOTIFY signatureChanged)

    Q_PROPERTY(QString domain READ domain WRITE setDomain NOTIFY domainChanged)
    Q_PROPERTY(QString serverAddress READ serverAddress WRITE setServerAddress NOTIFY serverAddressChanged)
    Q_PROPERTY(int port READ port WRITE setPort NOTIFY portChanged)
    Q_PROPERTY(bool secureConnection READ secureConnection WRITE setSecureConnection NOTIFY secureConnectionChanged)
    Q_PROPERTY(bool acceptUntrustedCertificates READ acceptUntrustedCertificates WRITE setAcceptUntrustedCertificates NOTIFY acceptUntrustedCertificatesChanged)
    Q_PROPERTY(QString certificate READ certificate WRITE setCertificate NOTIFY certificateChanged);
    Q_PROPERTY(QString certPassphrase READ certPassphrase WRITE setCertPassphrase NOTIFY certPassphraseChanged);

public:
    ActiveSyncAccountProvisioner(QObject *parent = nullptr);

    QString username() const;
    void setUsername(const QString &username);

    QString password() const;
    void setPassword(const QString &password);

    QString emailAddress() const;
    void setEmailAddress(const QString &emailAddress);

    QString description() const;
    void setDescription(const QString &description);

    QString signature() const;
    void setSignature(const QString &signature);

    QString domain() const;
    void setDomain(const QString &domain);

    QString serverAddress() const;
    void setServerAddress(const QString &serverAddress);

    int port() const;
    void setPort(int port);

    bool secureConnection() const;
    void setSecureConnection(bool secureConnection);

    bool acceptUntrustedCertificates() const;
    void setAcceptUntrustedCertificates(bool acceptUntrustedCertificates);

    QString certificate() const;
    void setCertificate(const QString &certificate);

    QString certPassphrase() const;
    void setCertPassphrase(const QString &certPassphrase);

    bool createAccount() Q_DECL_OVERRIDE;

    bool selectAccount(quint64 accountId) Q_DECL_OVERRIDE;
    bool commitChanges() Q_DECL_OVERRIDE;

Q_SIGNALS:
    void usernameChanged();
    void passwordChanged();
    void emailAddressChanged();
    void descriptionChanged();
    void signatureChanged();

    void domainChanged();
    void serverAddressChanged();
    void portChanged();
    void secureConnectionChanged();
    void acceptUntrustedCertificatesChanged();
    void certificateChanged();
    void certPassphraseChanged();

private:
    Sailfish::Mdm::Accounts::ActiveSyncAccountProvisionerPrivate *d_ptr;
    Q_DISABLE_COPY(ActiveSyncAccountProvisioner)
    Q_DECLARE_PRIVATE(ActiveSyncAccountProvisioner)

};

} // Accounts
} // Mdm
} // Sailfish

QDBusArgument &operator<<(QDBusArgument &, const Sailfish::Mdm::Accounts::Provisioner::AccountInfo &) Q_DECL_EXPORT;
const QDBusArgument &operator>>(const QDBusArgument &, Sailfish::Mdm::Accounts::Provisioner::AccountInfo &) Q_DECL_EXPORT;

Q_DECLARE_METATYPE(Sailfish::Mdm::Accounts::Provisioner::AccountInfo)
Q_DECLARE_METATYPE(QList<Sailfish::Mdm::Accounts::Provisioner::AccountInfo>)

#endif // SAILFISH_MDM_ACCOUNTS_H
