/*
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef BASE_MDM_PROXY_IFACE_H
#define BASE_MDM_PROXY_IFACE_H

#include <QDBusConnection>
#include <QDBusInterface>
#include <QDBusPendingCall>
#include <QDBusPendingCallWatcher>
#include <QDBusPendingReply>
#include <QDBusReply>
#include <QDebug>
#include <QObject>
#include <QString>

#include <functional>
#include <memory>

const auto UserSessionWarning = QStringLiteral(" module is not available before user session");

using ErrorInfo = QPair<QString,QString>;

namespace Sailfish {
namespace Mdm {

class UserManager;

} // namespace Mdm
} // namespace Sailfish


class BaseMdmProxyInterface : public QObject
{
    Q_OBJECT

public:
    enum ProxyType { User = 0, Root };
    Q_ENUM(ProxyType);

    BaseMdmProxyInterface(const QString &moduleName, ProxyType proxyType, QObject *parent = nullptr, bool waitUserSessionModules = true);
    virtual ~BaseMdmProxyInterface() = default;

    template<typename T, typename... Args>
    T call(const QString &method, T defaultValue, Args... args) const
    {
        if (m_proxyInterface == nullptr) {
            qWarning() << m_moduleName + UserSessionWarning;
            return defaultValue;
        }
        QDBusReply<T> reply = m_proxyInterface->call(method, std::forward<Args>(args)...);
        if (reply.isValid())
            return reply.value();
        qWarning() << m_moduleName << "error reply on method" << method << ":"
                   << reply.error().message();
        return defaultValue;
    }

    template<typename T, typename... Args>
    T callWithGetError(ErrorInfo &error, const QString &method, T defaultValue, Args... args) const
    {
        error.first.clear();
        error.second.clear();
        if (m_proxyInterface == nullptr) {
            error.second = m_moduleName + UserSessionWarning;
            qWarning() << error.second;
            return defaultValue;
        }
        QDBusReply<T> reply = m_proxyInterface->call(method, std::forward<Args>(args)...);
        if (reply.isValid())
            return reply.value();
        error.first = reply.error().name();
        error.second = reply.error().message();
        return defaultValue;
    }

    template <typename T> /* TODO move to more generic template logic */
    std::tuple<T, QString, int> accountCall(const QString &method, T defaultValue, const QVariantList &args) const
    {
        if (m_proxyInterface == nullptr) {
            qWarning() << m_moduleName + UserSessionWarning;
            return {defaultValue, m_moduleName + UserSessionWarning, 1};
        }

        QDBusPendingReply<T, QString, int> reply = m_proxyInterface->asyncCallWithArgumentList(method, args);
        reply.waitForFinished();
        if (!reply.isValid())
            return {defaultValue, reply.error().message(), 1};

        T value = reply.template argumentAt<0>();
        QString errorMessage = reply.template argumentAt<1>();
        int errorCode = reply.template argumentAt<2>();
        return {value, errorMessage, errorCode};
    }

    template <typename... Args>
    bool voidCall(const QString &method, Args... args) const
    {
        if (m_proxyInterface == nullptr) {
            qWarning() << m_moduleName + UserSessionWarning;
            return false;
        }
        QDBusReply<void> reply = m_proxyInterface->call(method, std::forward<Args>(args)...);
        if (reply.isValid())
            return true;
        qWarning() << m_moduleName << "error reply on method" << method << ":" << reply.error().message();
        return false;
    }

    template <typename... Args>
    QString voidCallWithGetError(const QString &method, Args... args) const
    {
        if (m_proxyInterface == nullptr)
            return m_moduleName + UserSessionWarning;
        QDBusReply<void> reply = m_proxyInterface->call(method, std::forward<Args>(args)...);
        if (reply.isValid())
            return {};
        return reply.error().message();
    }

    template<typename ResType, typename... Args>
    bool asyncCall(std::function<void(const QDBusPendingReply<ResType> &)> callback,
                   const QString &method, Args... args)
    {
        if (m_proxyInterface == nullptr) {
            qWarning() << m_moduleName + UserSessionWarning;
            return false;
        }
        QDBusPendingCall pcall = m_proxyInterface->asyncCall(method, std::forward<Args>(args)...);
        QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher(pcall, this);
        QObject::connect(watcher, &QDBusPendingCallWatcher::finished,
                         [callback, this, method](QDBusPendingCallWatcher *call) {
            QDBusPendingReply<ResType> reply = *call;
            if (reply.isError() || !reply.isValid())
                qWarning() << m_moduleName << "error reply on method" << method << ":" << reply.error().message();
            if (callback)
                callback(reply);
            call->deleteLater();
        });
        return true;
    }

    void connectToSignal(const QString &signal, const char *slot);
    void connectToSignal(const QString &signal, QObject *receiver, const char *slot);

    virtual void customInitProxyPrep();
    void scheduleProxyInterfaceInit();

    QString proxyServiceName(); 
    QString proxyObjectPath(); 
    QString proxyInterfaceName(); 
    void setTimeout(int);

    mutable QScopedPointer<QDBusInterface> m_proxyInterface;

private slots:
    void initProxyInterface();

private:
    QString m_moduleName;
    ProxyType m_proxyType;
    // This type of pointer dont use sizeof
    std::shared_ptr<Sailfish::Mdm::UserManager> m_userManager;
    bool m_waitUserSessionModules;
};

#endif // BASE_MDM_PROXY_IFACE_H
