/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013 Jolla Ltd
** SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.4
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0

SilicaMouseArea {
    id: slider

    property real maximumValue: 1.0
    property real minimumValue: 0.0
    property real stepSize
    property real value: 0.0
    readonly property real sliderValue: Math.max(minimumValue, Math.min(maximumValue, value))
    property bool handleVisible: true
    property string valueText
    property bool down: pressed && !DragFilter.canceled && !_cancel
    property real leftMargin: Math.round(Screen.width/8)
    property real rightMargin: Math.round(Screen.width/8)

    property int colorScheme: palette.colorScheme
    property color color: palette.highlightColor
    property color backgroundColor: palette.primaryColor
    property color highlightColor: Qt.tint(Theme.highlightColor, Theme.rgba(Theme.highlightBackgroundColor, 0.3))

    property color secondaryHighlightColor: palette.secondaryHighlightColor
    property color valueLabelColor: palette.primaryColor

    property real _oldValue
    property real _precFactor: 1.0

    property real _grooveWidth: Math.max(0, width - leftMargin - rightMargin)
    property bool _widthChanged
    property bool animateValue: true
    property bool _cancel

    property bool _componentComplete

    property Item _highlightItem
    property Item _backgroundItem
    property Item _progressBarItem

    readonly property real _lineLeftPoint: leftMargin
    readonly property real _lineRightPoint: width - rightMargin
    readonly property real _handleLeftPoint: _lineLeftPoint + _highlightItem.width * 0.5
    readonly property real _handleRightPoint: _lineRightPoint - _highlightItem.width * 0.5
    readonly property real _highlightX: handleVisible
                                        ? _handleHighlightX
                                        : leftMargin + (sliderValue - minimumValue) / (maximumValue - minimumValue)
                                          * (_lineRightPoint - _lineLeftPoint)

    property real _handleHighlightX: {
        if (maximumValue > minimumValue) {
            return leftMargin + (sliderValue - minimumValue) / (maximumValue - minimumValue)
                    * (_handleRightPoint - _handleLeftPoint)
        }

        return leftMargin
    }

    Behavior on _handleHighlightX {
        enabled: !_widthChanged && animateValue
        SmoothedAnimation {
            duration: 300
            velocity: Theme.dp(1500)
        }
    }

    property real _progressBarWidth: (sliderValue - minimumValue) / (maximumValue - minimumValue)
                                      * (_backgroundItem.width - _progressBarItem.height) + _progressBarItem.height

    Behavior on _progressBarWidth {
        enabled: !_widthChanged && animateValue
        SmoothedAnimation {
            duration: 300
            velocity: Theme.dp(1500)
        }
    }

    highlighted: down
    cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor

    DragFilter.orientations: Qt.Vertical
    onPreventStealingChanged: if (preventStealing) slider.DragFilter.end()

    onStepSizeChanged: {
        // Avoid rounding errors.  We assume that the range will
        // be sensibly related to stepSize
        var decimial = Math.floor(stepSize) - stepSize
        if (decimial < 0.001) {
            _precFactor = Math.pow(10, 7)
        } else if (decimial < 0.1) {
            _precFactor = Math.pow(10, 4)
        } else {
            _precFactor = 1.0
        }
    }

    onWidthChanged: updateWidth()
    onLeftMarginChanged: updateWidth()
    onRightMarginChanged: updateWidth()

    // changing the width of the slider shouldn't animate the slider bar/handle
    function updateWidth() {
        _widthChanged = true
        _grooveWidth = Math.max(0, width - leftMargin - rightMargin)
        _widthChanged = false
    }

    function cancel() {
        _cancel = true
        value = _oldValue
    }

    drag {
        target: draggable
        minimumX: leftMargin - _highlightItem.width * 0.5
        maximumX: slider.width - rightMargin - _highlightItem.width * 0.5
        axis: Drag.XAxis
        onActiveChanged: if (drag.active && !slider.DragFilter.canceled) slider.DragFilter.end()
        smoothed: false
    }

    function _updateValueToDraggable() {
        if (width > (leftMargin + rightMargin)) {
            const dragX = draggable.x + draggable.width * 0.5
            const rangeLength = _handleRightPoint - _handleLeftPoint
            const propgress = Math.min(Math.max(0.0, (dragX - _handleLeftPoint)) / rangeLength, 1.0)
            value = _calcValue(propgress * (maximumValue - minimumValue) + minimumValue)
        }
    }

    function _calcValue(newVal) {
        if (newVal <= minimumValue) {
            return minimumValue
        }

        if (stepSize > 0.0) {
            var offset = newVal - minimumValue
            var intervals = Math.round(offset / stepSize)
            newVal = Math.round((minimumValue + (intervals * stepSize)) * _precFactor) / _precFactor
        }

        if (newVal > maximumValue) {
            return maximumValue
        }

        return newVal
    }

    onPressed: {
        slider.DragFilter.begin(mouse.x, mouse.y)
        _cancel = false
        _oldValue = value
        draggable.x = Math.min(Math.max(drag.minimumX, mouseX - _highlightItem.width * 0.5), drag.maximumX)
    }

    onReleased: {
        if (!_cancel) {
            _updateValueToDraggable()
            _oldValue = value
        }
    }

    onCanceled: {
        slider.DragFilter.end()
        value = _oldValue
    }

    FontMetrics {
        id: fontMetrics
        font.pixelSize: Theme.fontSizeHuge
    }

    Item {
        id: draggable
        width: _highlightItem.width
        height: _highlightItem.height
        onXChanged: {
            if (_cancel || slider.DragFilter.canceled) {
                return
            }
            if (slider.drag.active) {
                _updateValueToDraggable()
            }
        }
    }

    states: State {
        name: "invalidRange"
        when: _componentComplete && minimumValue >= maximumValue
        PropertyChanges {
            target: _progressBarItem
            width: 0
        }
        PropertyChanges {
            target: slider
            enabled: false
            opacity: Theme.opacityHigh
        }
        StateChangeScript {
            script: console.log("Warning: Slider.maximumValue needs to be higher than Slider.minimumValue")
        }
    }

    Component.onCompleted: {
        _componentComplete = true
    }
}
