/*
 * SPDX-FileCopyrightText: 2018–2019 Jolla Ltd
 * SPDX-FileCopyrightText: 2019-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */
import QtQuick 2.5
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0

Page {
    id: page

    property var path
    property var archiveFile
    property var fileName
    property var baseExtractionDirectory
    property int __archive_page
    readonly property string __directory_path: extractor.isInternalMode ? archiveFile + (path !== "/" ? path : "")
                                                                        : "/" + fileName + (path !== "/" ? path : "")

    property string title
    property FileManagerNotification errorNotification
    property ExtractorView extractor

    signal archiveExtracted(string containingFolder)

    showNavigationIndicator: backNavigation

    Component.onCompleted: {
        if (!errorNotification) {
            errorNotification = errorNotificationComponent.createObject(page)
        }

        FileManager.init(pageStack)
    }

    onStatusChanged: {
        if (status == PageStatus.Activating) {
            extractor.path = page.path
            files.opacity = 1.0
            pathBar.updatePosition()
            archivePathModel.path = page.__directory_path.length > 0 ? page.__directory_path : ""
        }

        if (status == PageStatus.Deactivating) {
            files.opacity = 0.0
        }
    }

    FileGridHeader {
        id: appBar

        headerTitle: page.title.length > 0 ? page.title
                                           : path !== "/" ? extractor.model.currentDirName : fileName
        parentGridView: files
        path: extractor.model.path
        countFiles: extractor.model.count
        errorType: extractor.model.errorState
        isArchiveMode: true
        z: page.z + 1

        onExtractArchive: {
            extractor.rejectedPageName = "__archive_page"
            extractor.selectedExtractPath(extractor.extractAllFiles)
        }
    }

    PathBar {
        id: pathBar

        anchors {
            top: appBar.bottom
            topMargin: Theme.paddingMedium
        }

        z: appBar.z
        model: archivePathModel
        enabled: !busyIndicator.running
        visible: enabled
        homeButtonVisible: extractor.isInternalMode
        homeButtonIcon.source: homeButtonVisible ? "image://theme/icon-splus-home"
                                                 : "image://theme/icon-splus-extract-out"

    }

    DirectoryPathModel {
        id: archivePathModel

        archiveMode: true
    }

    SilicaListView {
        id: files

        readonly property bool archiveMode: true

        objectName: "ArchivePageSilicaListView"
        anchors {
            fill: parent
            topMargin: appBar.height + pathBar.height
        }
        clip: true
        delegate: InternalFileItem {
            id: delegateItem

            function cleanUp() {
                //% "Deleted extracted directory"
                var text = model.isDir ? qsTrId("filemanager-la-deleted_extracted_dir")
                                       : //% "Deleted extracted file"
                                         qsTrId("filemanager-la-deleted_extracted_file")
                remorseAction(text, function() {
                    FileEngine.deleteFiles(model.extractedTargetPath, true)
                    extractor.model.cleanExtractedEntry(model.fileName)
                })
            }

            width: ListView.view.width
            menu: contextMenu
            compressed: true

            onClicked: {
                if (delegateItem.handleMouseRightClick(mouse))
                    return

                if (model.isDir) {
                    var obj = FileManager.openArchive(archiveFile, extractor.appendPath(model.fileName), baseExtractionDirectory, extractor, false)
                    obj.pageCompleted.connect(function(archivePage) {
                        archivePage.archiveExtracted.connect(page.archiveExtracted)
                    })
                } else {
                    openMenu()
                }
            }

            Component {
                id: contextMenu

                ContextMenu {
                    ExtractContextMenuItem {
                        objectName: "ExtractHereMenuItem"

                        extractorView: extractor
                        visible: extractor.extractHereAvailable
                        destinationPageName: "__archive_page"
                        rejectedPageName: destinationPageName
                    }

                    ExtractContextMenuItem {
                        objectName: "ExtractToMenuItem"

                        extractorView: extractor
                        needSelectDirectory: true
                        destinationPageName: "__archive_page"
                        rejectedPageName: destinationPageName
                    }
                }
            }
        }

        model: extractor.model

        Behavior on opacity { FadeAnimator { duration: 100 } }

        VerticalScrollDecorator {}

        Component {
            id: errorNotificationComponent

            FileManagerNotification {}
        }

        PageBusyIndicator {
            id: busyIndicator

            running: extractor.model.status < ArchiveModel.Ready

            z: 1
            FadeAnimator {
                target: busyIndicator
                duration: 1000
                easing.type: Easing.InExpo
                to: 1.0
            }
        }
    }

    NoFilesPagePlaceholder {
        objectName: "NoFilesPlaceholder"
        visible: extractor.model.count === 0 && !busyIndicator.running

        label.text: !FileManager.openArchiveIsCorrupted
                      //% "No files"
                    ? qsTrId("filemanager-la-no_files")
                    : qsTrId("filemanager-la-opening_archive_failed")
    }
}
