/****************************************************************************************
**
** SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: Proprietary
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Aurora.Controls 1.0

SilicaMouseArea {
    id: button

    property bool down: pressed && containsMouse
    property alias text: buttonText.text
    property color color: palette.primaryColor
    property real preferredWidth: Theme.iconSizeMedium
    property color highlightColor: palette.highlightColor
    property alias icon: image
    property alias layoutDirection: content.layoutDirection
    property string context
    property bool _highlightedByAppBar

    onVisibleChanged: if (parent) parent._updateComponentsWidth()
    onWidthChanged: if (parent) parent._updateComponentsWidth()

    onPressAndHold: if (parent && context) parent._setContext(button.context)
    onReleased: parent._resetContext()
    onCanceled: parent._resetContext()

    onEntered: {
        if (parent && context && !pressed) {
            if (parent._contextVisible) {
                if (parent && context) parent._setContext(button.context)
            } else {
                hoverTimer.start()
            }
        }
    }
    onExited: {
        parent._resetContextNow()
        hoverTimer.stop()
    }
    onPressed: {
        parent._resetContextNow()
        hoverTimer.stop()
    }

    height: implicitHeight
    implicitHeight: Theme.dp(72)
    implicitWidth: Math.max(preferredWidth, content.implicitWidth)

    hoverEnabled: true
    cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor

    Timer {
        id: hoverTimer
        interval: 1000
        onTriggered: if (button.parent && button.context) button.parent._setContext(button.context)
    }

    Row {
        id: content

        anchors.centerIn: parent
        opacity: button.enabled ? 1.0 : Theme.opacityLow

        Item {
            anchors.verticalCenter: parent.verticalCenter
            width: Theme.dp(72)
            height: Theme.dp(72)

            visible: image.source != ""

            Icon {
                id: image

                anchors.centerIn: parent
                sourceSize: Qt.size(Theme.iconSizeSmallPlus, Theme.iconSizeSmallPlus)
                color: down || button.highlighted || button._highlightedByAppBar ? button.highlightColor : button.color
            }
        }

        Label {
            id: buttonText

            anchors.verticalCenter: parent.verticalCenter
            visible: text != ""

            leftPadding: {
                if (image.width == 0 || layoutDirection == Qt.RightToLeft) {
                    return Theme.horizontalPageMargin / 2
                }
                return 0
            }

            rightPadding: {
                if (image.width == 0 || layoutDirection == Qt.LeftToRight) {
                    return Theme.horizontalPageMargin / 2
                }
                return 0
            }

            color: down || button.highlighted || button._highlightedByAppBar ? button.highlightColor : button.color
            font.pixelSize: Theme.fontSizeSmall
            truncationMode: TruncationMode.Fade
        }
    }
}
