/*
 * Copyright (c) 2017 - 2020 Jolla Ltd.
 *
 * License: Proprietary
 */

#ifndef SAILFISH_MDM_SERIALISATION_IMPL_P_H
#define SAILFISH_MDM_SERIALISATION_IMPL_P_H

#include <QVariantList>
#include <QVariantMap>
#include <QVariant>
#include <QString>

#include <QDBusArgument>
#include <QDBusMetaType>

#include <QtDebug>

namespace {
    // forward declaration.
    bool expandVariantMap(const QVariantMap &map, QVariantMap *expandedMap, int depth);

    // QtDBus doesn't deserialise nested QVariantLists (or any form of complex type nested within a QVariantList)
    bool expandVariantList(const QVariantList &list, QVariantList *expandedList, int depth)
    {
        int newDepth = depth + 1;
        if (newDepth >= 5) {
            return false; // too many recursions.
        }

        QVariantList::const_iterator it = list.constBegin();
        while (it != list.constEnd()) {
            if (qstrcmp((*it).typeName(), "QDBusArgument") != 0) {
                // this value doesn't need expansion, append it directly
                expandedList->append(*it);
            } else {
                QDBusArgument dbusArg = (*it).value<QDBusArgument>();
                if (dbusArg.currentType() == QDBusArgument::MapType
                        && dbusArg.currentSignature() == QLatin1String("a{sv}")) {
                    QVariantMap mapValue = qdbus_cast<QVariantMap>(dbusArg);
                    QVariantMap expandedMapValue;
                    if (expandVariantMap(mapValue, &expandedMapValue, newDepth)) {
                        expandedList->append(expandedMapValue);
                    } else {
                        qWarning() << "Unable to expand map value in list";
                        return false;
                    }
                } else if (dbusArg.currentType() == QDBusArgument::ArrayType
                        && (dbusArg.currentSignature() == QLatin1String("av")
                            || dbusArg.currentSignature() == QLatin1String("a(v)"))) {
                    QVariantList listValue = qdbus_cast<QVariantList>(dbusArg);
                    QVariantList expandedListValue;
                    if (expandVariantList(listValue, &expandedListValue, newDepth)) {
                        expandedList->append(expandedListValue);
                    } else {
                        qWarning() << "Unable to expand list value in list";
                        return false;
                    }
                } else {
                    qWarning() << "Don't know how to expand type:" << dbusArg.currentType()
                               << "in list with signature:" << dbusArg.currentSignature();
                    return false;
                }
            }
            it++;
        }
        return true;
    }

    // QtDBus doesn't deserialise nested QVariantMaps (or any form of complex type nested within a QVariantMap)
    bool expandVariantMap(const QVariantMap &map, QVariantMap *expandedMap, int depth)
    {
        int newDepth = depth + 1;
        if (newDepth >= 5) {
            qWarning() << "Too many recursions, not expanding map at depth" << depth;
            return false;
        }

        QVariantMap::const_iterator it = map.constBegin();
        while (it != map.constEnd()) {
            if (qstrcmp(it.value().typeName(), "QDBusArgument") != 0) {
                // this value doesn't need expansion, insert it directly.
                expandedMap->insert(it.key(), it.value());
            } else {
                // this value needs expansion.
                QDBusArgument dbusArg = it.value().value<QDBusArgument>();
                if (dbusArg.currentType() == QDBusArgument::MapType
                        && dbusArg.currentSignature() == QLatin1String("a{sv}")) {
                    QVariantMap mapValue = qdbus_cast<QVariantMap>(dbusArg);
                    QVariantMap expandedMapValue;
                    if (expandVariantMap(mapValue, &expandedMapValue, newDepth)) {
                        expandedMap->insert(it.key(), expandedMapValue);
                    } else {
                        qWarning() << "Unable to expand map value for key:" << it.key();
                        return false;
                    }
                } else if (dbusArg.currentType() == QDBusArgument::ArrayType
                           && (dbusArg.currentSignature() == QLatin1String("av")
                               || dbusArg.currentSignature() == QLatin1String("a(v)"))) {
                    QVariantList listValue = qdbus_cast<QVariantList>(dbusArg);
                    QVariantList expandedListValue;
                    if (expandVariantList(listValue, &expandedListValue, newDepth)) {
                        expandedMap->insert(it.key(), expandedListValue);
                    } else {
                        qWarning() << "Unable to expand list value for key:" << it.key();
                        return false;
                    }
                } else {
                    qWarning() << "Don't know how to expand type:" << dbusArg.currentType()
                               << "for key:" << it.key() << "with signature:" << dbusArg.currentSignature();
                    return false;
                }
            }
            it++;
        }
        return true;
    }
}

#endif // SAILFISH_MDM_SERIALISATION_IMPL_P_H
