/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013 Jolla Ltd
** SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import "private"

// TODO: Investigate using a scaled down version of the image

SilicaItem {
    id: timePicker

    property int hour
    property int minute
    property int _second

    property int hourMode: DateTime.DefaultHours

    property date time: new Date(0,0,0, hour, minute, _second)
    property string timeText: _formatTime(hour, minute, _second)

    property int _mode: TimePickerMode.HoursAndMinutes
    readonly property bool _hoursAndMinutes: _mode === TimePickerMode.HoursAndMinutes

    QtObject {
        id: internal

        readonly property int innerWatchfaceWidth: Theme.dp(266)
        readonly property int outerWatchfaceWidth: Theme.dp(406)
        readonly property int tickHeight: Theme.dp(10)
    }

    width: Theme.dp(490)
    height: width

    onHourChanged: {
        hour = (hour < 0 ? 0 : (hour > 23 ? 23 : hour))
        _updateHourIndicator()
    }

    onMinuteChanged: {
        minute = (minute < 0 ? 0 : (minute > 59 ? 59 : minute))
        _updateMinuteIndicator()
    }

    on_SecondChanged: {
        _second = (_second < 0 ? 0 : (_second > 59 ? 59 : _second))
        _updateSecondIndicator()
    }

    on_HoursAndMinutesChanged: {
        if (_hoursAndMinutes) {
            _updateHourIndicator()
            _updateMinuteIndicator()
        } else {
            _updateMinuteIndicator()
            _updateSecondIndicator()
        }
    }

    function _updateHourIndicator() {
        if (mouse.changingProperty == 0 && _hoursAndMinutes) {
            var delta = (hour - innerIndicator.value)
            innerIndicator.value += (delta % 24)
        }
    }

    function _updateMinuteIndicator() {
        if (mouse.changingProperty == 0) {
            if (_hoursAndMinutes) {
                var delta = (minute - outerIndicator.value)
                outerIndicator.value += (delta % 60)
            } else {
                var delta = (minute - innerIndicator.value)
                innerIndicator.value += (delta % 60)
            }
        }
    }

    function _updateSecondIndicator() {
        if (mouse.changingProperty == 0 && !_hoursAndMinutes) {
            var delta = (_second - outerIndicator.value)
            outerIndicator.value += (delta % 60)
        }
    }

    function _formatTime(hour, minute, _second) {
        var date = new Date()
        date.setSeconds(_second)
        date.setHours(hour)
        date.setMinutes(minute)
        var format
        if (_hoursAndMinutes) {
            format = (hourMode == DateTime.DefaultHours ? Formatter.TimeValue
                                                        : (hourMode == DateTime.TwentyFourHours ? Formatter.TimeValueTwentyFourHours
                                                                                                : Formatter.TimeValueTwelveHours))
        } else {
            format = Formatter.DurationShort
        }
        return Format.formatDate(date, format)
    }

    ShaderEffect {
        readonly property color rColor: timePicker.palette.primaryColor
        readonly property color gColor: Theme.rgba(timePicker.palette.primaryColor, Theme.opacityLow)
        readonly property color bColor: "transparent"

        readonly property real exclusion1RotationRadius: innerIndicator.rotationRadius / width
        readonly property real exclusion1RotationAngle: innerIndicator.angle
        readonly property real exclusion1Radius: 0.5 * innerIndicator.width / width

        readonly property real exclusion2RotationRadius: outerIndicator.rotationRadius / width
        readonly property real exclusion2RotationAngle: outerIndicator.angle
        readonly property real exclusion2Radius: 0.5 * outerIndicator.width / width

        readonly property Image source: Image {
            width: internal.outerWatchfaceWidth * 0.5
            height: width
            visible: false
            source: _hoursAndMinutes ? "private/watchface-hr-min.svg" : "private/watchface-min-sec.svg"
        }

        anchors.centerIn: parent
        width: internal.outerWatchfaceWidth
        height: width

        fragmentShader: "
            #define PI2 1.57079632679 // pi / 2

            precision lowp float;

            varying highp vec2 qt_TexCoord0;
            uniform sampler2D source;
            uniform float qt_Opacity;

            uniform vec4 rColor;
            uniform vec4 gColor;
            uniform vec4 bColor;

            uniform float exclusion1RotationRadius;
            uniform float exclusion1RotationAngle;
            uniform float exclusion1Radius;

            uniform float exclusion2RotationRadius;
            uniform float exclusion2RotationAngle;
            uniform float exclusion2Radius;

            // Rotates a texture coordinate around the center by 90-degree steps
            highp vec2 rotate90(highp vec2 coord, float turns) {
                vec2 pivot = vec2(0.5);
                vec2 offset = coord - pivot;
                float angle = PI2 * turns;
                float c = cos(angle);
                float s = sin(angle);
                return pivot + vec2(offset.x * c - offset.y * s,
                                    offset.x * s + offset.y * c);
            }

            void main() {
                vec2 coord = qt_TexCoord0;
                vec2 doubled = coord * 2.0;
                vec2 texCoord;

                // Determine which quadrant we're in and rotate accordingly
                if (coord.x <= 0.5 && coord.y <= 0.5) {
                    texCoord = doubled;
                } else if (coord.x > 0.5 && coord.y <= 0.5) {
                    texCoord = rotate90(doubled - vec2(1.0, 0.0), -1.0);
                } else if (coord.x <= 0.5 && coord.y > 0.5) {
                    texCoord = rotate90(doubled - vec2(0.0, 1.0), 1.0);
                } else {
                    texCoord = rotate90(doubled - vec2(1.0, 1.0), 2.0);
                }

                vec4 src = texture2D(source, texCoord);

                // Skip fully transparent pixels
                if (src.a < 0.01) {
                    gl_FragColor = vec4(0.0);
                    return;
                }

                // Position relative to center of the texture
                vec2 center = coord - vec2(0.5);

                // Centers of the exclusion zones
                vec2 exclusion1 = vec2(exclusion1RotationRadius * cos(exclusion1RotationAngle - PI2), exclusion1RotationRadius * sin(exclusion1RotationAngle - PI2));
                vec2 exclusion2 = vec2(exclusion2RotationRadius * cos(exclusion2RotationAngle - PI2), exclusion2RotationRadius * sin(exclusion2RotationAngle - PI2));

                // Skip if inside any exclusion zone
                if (dot(center - exclusion1, center - exclusion1) < pow(exclusion1Radius, 2.0) ||
                    dot(center - exclusion2, center - exclusion2) < pow(exclusion2Radius, 2.0)) {
                    gl_FragColor = vec4(0.0);
                    return;
                }

                // Blend the sampled RGB values with provided colors
                vec4 color = src.r * rColor + src.g * gColor + src.b * bColor;
                gl_FragColor = min(color, vec4(1.0)) * src.a * qt_Opacity;
            }
        "
    }

    TimePickerHandle {
        id: innerIndicator

        anchors.centerIn: parent
        stepCount: _hoursAndMinutes ? 24 : 60
        rotationRadius: (internal.innerWatchfaceWidth - internal.tickHeight) * 0.5
        velocity: _hoursAndMinutes ? 30 : 80
        highlighted: mouse.changingProperty == 1
        moving: mouse.isMoving && !mouse.isLagging
        palette: timePicker.palette
    }

    TimePickerHandle {
        id: outerIndicator

        anchors.centerIn: parent
        stepCount: 60
        rotationRadius: (internal.outerWatchfaceWidth - internal.tickHeight) * 0.5
        velocity: 80
        highlighted: mouse.changingProperty == 2
        moving: mouse.isMoving && !mouse.isLagging
        palette: timePicker.palette
    }

    MouseArea {
        id: mouse

        property int changingProperty
        property bool isMoving
        property bool isLagging

        anchors.fill: parent
        preventStealing: true
        cursorShape: enabled ? Qt.PointingHandCursor : Qt.ArrowCursor

        function radiusForCoord(x, y) {
            // Return the distance from the mouse position to the center
            return Math.sqrt(Math.pow(x, 2) + Math.pow(y, 2))
        }

        function angleForCoord(x, y) {
            // Return the angular position in degrees, rising anticlockwise from the positive X-axis
            var result = Math.atan(y / x) / (Math.PI * 2) * 360

            // Adjust for various quadrants
            if (x < 0)  {
                result += 180
            } else if (y < 0) {
                result += 360
            }
            return result
        }

        function remapAngle(value, bound) {
            // Return the angle in degrees mapped to the adjusted range 0 - (bound-1) and
            // translated to the clockwise from positive Y-axis orientation
            return Math.round(bound - (((value - 90) / 360) * bound)) % bound
        }

        function remapMouse(mouseX, mouseY) {
            // Return the mouse coordinates in cartesian coords relative to the circle center
            return { x: mouseX - (width / 2), y: 0 - (mouseY - (height / 2)) }
        }

        function propertyForRadius(radius) {
            // Return the property associated with clicking at radius distance from the center
            if (radius < (internal.innerWatchfaceWidth * 0.5 + innerIndicator.width)) {
                return 1 // Hours
            } else if (radius < (internal.outerWatchfaceWidth * 0.5 + outerIndicator.width)) {
                return 2 // Minutes
            }
            return 0
        }

        function updateForAngle(angle) {
            // Update the selected property for the specified angular position
            var indicator = changingProperty == 1 ? innerIndicator : outerIndicator
            var stepCount = indicator.stepCount
            var value = remapAngle(angle, stepCount)

            // Round single touch to the nearest 5 min/second mark
            if (stepCount > 24 && !isMoving) value = (Math.round(value / 5) * 5) % stepCount
            var delta = (value - indicator.value) % stepCount

            // It is not possible to make jumps of more than half of stepCount - reverse the direction
            if (delta > stepCount * 0.5) {
                delta -= stepCount
            } else if (delta < -stepCount * 0.5) {
                delta += stepCount
            }
            if (isMoving && isLagging) {
                if (Math.abs(delta) < 0.5) {
                    isLagging = false
                }
            }

            var target = (indicator.value + delta)
            indicator.value += delta

            if (changingProperty == 1 && _hoursAndMinutes) {
                if (target < 0) {
                    var multiple = Math.ceil(target / 24)
                    target += ((-multiple + 1) * 24)
                }
                timePicker.hour = (target % 24)
            } else if (changingProperty == 2 && !_hoursAndMinutes) {
                timePicker._second = value
            } else {
                timePicker.minute = value
            }
        }

        onPressed: {
            var coords = remapMouse(mouseX, mouseY)
            var radius = radiusForCoord(coords.x, coords.y)

            changingProperty = propertyForRadius(radius)
            if (changingProperty != 0) {
                preventStealing = true
                var angle = angleForCoord(coords.x, coords.y)

                isLagging = true
                updateForAngle(angle)
            } else {
                // Outside the minutes band - allow pass through to underlying component
                preventStealing = false
            }
        }

        onPositionChanged: {
            if (changingProperty > 0) {
                var coords = remapMouse(mouseX, mouseY)
                var angle = angleForCoord(coords.x, coords.y)

                isMoving = true
                updateForAngle(angle)
            }
        }

        onReleased: {
            changingProperty = 0
            isMoving = false
            isLagging = false
        }
    }
}
