/**
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef AURORA_MDM_MANAGEDCONFIGURATION_SERIALISATION_IMPL_P_H
#define AURORA_MDM_MANAGEDCONFIGURATION_SERIALISATION_IMPL_P_H

#include "mdm-managedconfiguration_p.h"

#include <QDBusArgument>
#include <QDebug>

using namespace Aurora::Mdm;
using Type = ManagedConfigurationValue::Type;
using ConfigurationType = ManagedConfigurationValue::ConfigurationType;
using Reason = ManagedConfigurationValue::Reason;
using Detail = ManagedConfigurationValue::Detail;
using Option = ManagedConfigurationValue::Option;
using ManagedConfigurationValueMap = QMap<QString, ManagedConfigurationValue>;

#define MC_TOKEN_VALUE "value"
#define MC_TOKEN_DEFAULT_VALUE "defaultValue"
#define MC_TOKEN_TYPE "type"
#define MC_TOKEN_TITLE "title"
#define MC_TOKEN_DESCRIPTION "description"
#define MC_TOKEN_CONFIGURABLE_BY "configurableBy"
#define MC_TOKEN_REASON "reason"
#define MC_TOKEN_OPTIONS "options"

const QDBusArgument &operator>>(const QDBusArgument &arg,
                                ManagedConfigurationValue &configurationValue);
const QDBusArgument &operator>>(const QDBusArgument &arg, Detail &detail);
const QDBusArgument &operator>>(const QDBusArgument &arg, Option &option);
const QDBusArgument &operator>>(const QDBusArgument &arg,
                                ManagedConfigurationValueMap &configuration);
const QDBusArgument &operator>>(const QDBusArgument &arg, ManagedConfiguration &configuration);

QDBusArgument &operator<<(QDBusArgument &arg, const ManagedConfigurationValue &configurationValue);
QDBusArgument &operator<<(QDBusArgument &arg, const Detail &detail);
QDBusArgument &operator<<(QDBusArgument &arg, const Option &option);
QDBusArgument &operator<<(QDBusArgument &arg, const ManagedConfigurationValueMap &configuration);
QDBusArgument &operator<<(QDBusArgument &arg, const ManagedConfiguration &configuration);

namespace {

QVariant parseOptions(const QVariant &variant)
{
    QList<Option> options;

    if (variant.type() != QVariant::Type::UserType
        || variant.userType() != qMetaTypeId<QDBusArgument>()) {
        qWarning() << "Unexpected type: expect" << QMetaType::typeName(qMetaTypeId<QDBusArgument>())
                   << "get" << QMetaType::typeName(variant.userType());
        return QVariant::fromValue(options);
    }

    const QDBusArgument &arg = variant.value<QDBusArgument>();
    arg >> options;
    return QVariant::fromValue(options);
}

template<typename T>
QVariant parseArrayValue(const QVariant &variant)
{
    QList<T> list;

    if (variant.type() != QVariant::Type::UserType
        || variant.userType() != qMetaTypeId<QDBusArgument>()) {
        qWarning() << "Unexpected type: expect" << QMetaType::typeName(qMetaTypeId<QDBusArgument>())
                   << "get" << QMetaType::typeName(variant.userType());
        return QVariant::fromValue(list);
    }

    const QDBusArgument &arg = variant.value<QDBusArgument>();
    arg >> list;
    return QVariant::fromValue(list);
}

QVariant parseStructureValue(const QVariant &variant)
{
    QMap<QString, ManagedConfigurationValue> structureData;

    if (variant.type() != QVariant::Type::UserType
        || variant.userType() != qMetaTypeId<QDBusArgument>()) {
        qWarning() << "Unexpected type: expect" << QMetaType::typeName(qMetaTypeId<QDBusArgument>())
                   << "get" << QMetaType::typeName(variant.userType());
        return QVariant::fromValue(structureData);
    }

    const QDBusArgument &arg = variant.value<QDBusArgument>();

    arg.beginMap();
    while (!arg.atEnd()) {
        QString key;
        QDBusVariant dbusVariant;
        ManagedConfigurationValue value;

        arg.beginMapEntry();
        arg >> key >> dbusVariant;
        arg.endMapEntry();

        QVariant variant = dbusVariant.variant();
        const QDBusArgument &varg = variant.value<QDBusArgument>();
        varg >> value;
        structureData.insert(key, value);
    }
    arg.endMap();

    return QVariant::fromValue(structureData);
};

QVariant parseValue(Type type, const QVariant &variant)
{
    switch (type) {
    case Type::Integer:     // int
    case Type::Bool:        // bool
    case Type::String:      // QString
    case Type::Select:      // QString
    case Type::MultiSelect: // QStringList
    case Type::StringArray: // QStringList
        return variant;
    case Type::IntegerArray:
        return parseArrayValue<int>(variant);
    case Type::BoolArray:
        return parseArrayValue<bool>(variant);
    case Type::Structure:
        return parseStructureValue(variant);
    case Type::Unknown:
        qWarning() << "Unknown type" << variant;
        return QVariant();
    default:
        qWarning() << "Unprocessed type:" << static_cast<int>(type);
        return QVariant();
    }
}

QVariant parseVariantData(Type type, Detail detail, const QVariant &variant)
{
    switch (detail) {
    case Detail::Title:
    case Detail::Description:
        return variant; // QString
    case Detail::Value:
    case Detail::DefaultValue:
        return parseValue(type, variant);
    case Detail::Type:
        return QVariant::fromValue(static_cast<Type>(variant.toInt()));
    case Detail::ConfigurationType:
        return QVariant::fromValue(static_cast<ConfigurationType>(variant.toInt()));
    case Detail::Reason:
        return QVariant::fromValue(static_cast<Reason>(variant.toInt()));
    case Detail::Options:
        return parseOptions(variant);
    case Detail::Unknown:
        return QVariant();
    default:
        qWarning() << "Unprocessed detail:" << static_cast<int>(detail);
        return QVariant();
    }
}

Detail detailFromString(const QString &detail)
{
    if (detail == MC_TOKEN_VALUE)
        return Detail::Value;
    if (detail == MC_TOKEN_DEFAULT_VALUE)
        return Detail::DefaultValue;
    if (detail == MC_TOKEN_TITLE)
        return Detail::Title;
    if (detail == MC_TOKEN_DESCRIPTION)
        return Detail::Description;
    if (detail == MC_TOKEN_TYPE)
        return Detail::Type;
    if (detail == MC_TOKEN_CONFIGURABLE_BY)
        return Detail::ConfigurationType;
    if (detail == MC_TOKEN_REASON)
        return Detail::Reason;
    if (detail == MC_TOKEN_OPTIONS)
        return Detail::Options;

    qWarning() << "Unknown detail" << detail;
    return Detail::Unknown;
}

} // namespace

const QDBusArgument &operator>>(const QDBusArgument &arg, Detail &detail)
{
    QString str;
    arg >> str;
    detail = detailFromString(str);
    return arg;
}

const QDBusArgument &operator>>(const QDBusArgument &arg, Option &option)
{
    arg.beginStructure();
    arg >> option.value >> option.description;
    arg.endStructure();
    return arg;
}

const QDBusArgument &operator>>(const QDBusArgument &arg, QList<Option> &options)
{
    arg.beginArray();
    while (!arg.atEnd()) {
        Option option;
        arg >> option;
        options.append(option);
    }
    arg.endArray();

    return arg;
}

const QDBusArgument &operator>>(const QDBusArgument &arg,
                                ManagedConfigurationValue &configurationValue)
{
    Type type = Type::Unknown;
    QMap<Detail, QVariant> unprocessedDetails;
    ManagedConfigurationValuePrivate *value = new ManagedConfigurationValuePrivate();

    // Lookup for type
    arg.beginMap();
    while (!arg.atEnd()) {
        Detail detail;
        QDBusVariant dbusVariant;

        arg.beginMapEntry();
        arg >> detail >> dbusVariant;
        arg.endMapEntry();

        QVariant variant = dbusVariant.variant();
        if (detail == Detail::Type) {
            QVariant typeDetail = parseVariantData(Type::Unknown, detail, variant);
            type = typeDetail.value<Type>();
            value->addDetail(detail, typeDetail);
        } else {
            unprocessedDetails.insert(detail, variant);
        }
    }
    arg.endMap();

    // Process other details when type is known
    for (auto it = unprocessedDetails.constBegin(); it != unprocessedDetails.constEnd(); ++it)
        value->addDetail(it.key(), parseVariantData(type, it.key(), it.value()));

    // ManagedConfigurationValuePrivate is moved to the ManagedConfigurationValue
    configurationValue = value->toManagedConfigurationValue();
    return arg;
}

const QDBusArgument &operator>>(const QDBusArgument &arg,
                                ManagedConfigurationValueMap &configuration)
{
    arg.beginMap();
    while (!arg.atEnd()) {
        QString key;
        ManagedConfigurationValue value;
        arg.beginMapEntry();
        arg >> key >> value;
        arg.endMap();
        configuration.insert(key, value);
    }

    arg.endMap();
    return arg;
}

const QDBusArgument &operator>>(const QDBusArgument &arg, ManagedConfiguration &configuration)
{
    arg.beginMap();
    while (!arg.atEnd()) {
        QString key;
        QDBusVariant dbusVariant;
        ManagedConfigurationValue value;

        arg.beginMapEntry();
        arg >> key >> dbusVariant;
        arg.endMapEntry();

        QVariant variant = dbusVariant.variant();
        const QDBusArgument &varg = variant.value<QDBusArgument>();
        varg >> value;
        configuration.add(key, value);
    }
    arg.endMap();
    return arg;
}

QDBusArgument &operator<<(QDBusArgument &arg, const Option &option)
{
    arg.beginStructure();
    arg << option.value << option.description;
    arg.endStructure();
    return arg;
}

QDBusArgument &operator<<(QDBusArgument &arg, const ManagedConfigurationValue &configurationValue)
{
    Type type = configurationValue.type();
    QVariant variant = configurationValue.value();

    switch (type) {
    case Type::Integer:
        arg << variant.toInt();
        break;
    case Type::String:
    case Type::Select:
        arg << variant.toString();
        break;
    case Type::Bool:
        arg << variant.toBool();
        break;
    case Type::IntegerArray:
        arg << variant.value<QList<int>>();
        break;
    case Type::StringArray:
    case Type::MultiSelect:
        arg << variant.value<QList<QString>>();
        break;
    case Type::BoolArray:
        arg << variant.value<QList<bool>>();
        break;
    case Type::Structure:
        arg << variant.value<QMap<QString, ManagedConfigurationValue>>();
        break;
    case Type::Unknown:
        qWarning() << "Skipping serialization of unknown type";
        break;
    default:
        qWarning() << "Unprocessed type: " << static_cast<int>(type);
        break;
    }

    return arg;
}

QDBusArgument &operator<<(QDBusArgument &arg, const Detail &detail)
{
    arg << static_cast<int>(detail);
    return arg;
}

QDBusArgument &operator<<(QDBusArgument &arg, const ManagedConfigurationValueMap &configuration)
{
    arg.beginMap(QMetaType::QString, qMetaTypeId<QDBusVariant>());
    for (const auto &key : configuration.keys()) {
        arg.beginMapEntry();
        QDBusArgument variant;
        variant << configuration[key];
        arg << key << QDBusVariant(QVariant::fromValue(variant));
        arg.endMapEntry();
    }
    arg.endMap();

    return arg;
}

QDBusArgument &operator<<(QDBusArgument &arg, const ManagedConfiguration &configuration)
{
    arg.beginMap(QMetaType::QString, qMetaTypeId<QDBusVariant>());
    for (const auto &key : configuration.keys()) {
        arg.beginMapEntry();
        QDBusArgument variant;
        variant << configuration.get(key);
        arg << key << QDBusVariant(QVariant::fromValue(variant));
        arg.endMapEntry();
    }
    arg.endMap();

    return arg;
}

#endif // AURORA_MDM_MANAGEDCONFIGURATION_SERIALISATION_IMPL_P_H
