/**
 * SPDX-FileCopyrightText: 2017-2020 Jolla Ltd.
 * SPDX-FileCopyrightText: 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef SAILFISH_MDM_LOCATIONSETTINGS_H
#define SAILFISH_MDM_LOCATIONSETTINGS_H

#include <QObject>
#include <QString>
#include <QDateTime>

namespace Sailfish {
namespace Mdm {

#define NOT_AVAILABLE_MSG   "Method is no more available, it returns default value"
#define DEPRECATED_MSG      "Method is deprecated. Use liblocation-settings-qt5 instead"

class LocationInfoPrivate;
class Q_DECL_EXPORT LocationInfo : public QObject
{
    Q_OBJECT
    Q_PROPERTY(double latitude READ latitude NOTIFY positionChanged)
    Q_PROPERTY(double longitude READ longitude NOTIFY positionChanged)
    Q_PROPERTY(double altitude READ altitude NOTIFY positionChanged)
    Q_PROPERTY(QDateTime timestamp READ timestamp NOTIFY positionChanged)
    Q_PROPERTY(Position position READ position NOTIFY positionChanged)
    Q_PROPERTY(int satellitesInUse READ satellitesInUse NOTIFY satellitesChanged)
    Q_PROPERTY(int satellitesInView READ satellitesInView NOTIFY satellitesChanged)

public:
    struct Position {
        Position(double lat = 0, double lon = 0, double alt = 0, const QDateTime &ts = QDateTime())
            : latitude(lat), longitude(lon), altitude(alt), timestamp(ts) {}
        double latitude;
        double longitude;
        double altitude;
        QDateTime timestamp;
    };

    LocationInfo(QObject *parent = Q_NULLPTR);
    ~LocationInfo();

    void startUpdates();
    void stopUpdates();
    void requestUpdate();

    double latitude() const;
    double longitude() const;
    double altitude() const;
    QDateTime timestamp() const;
    Position position() const;

    int satellitesInUse() const;
    int satellitesInView() const;

signals:
    void positionChanged();
    void satellitesChanged();

private:
    LocationInfoPrivate *d_ptr;
    Q_DISABLE_COPY(LocationInfo)
    Q_DECLARE_PRIVATE(LocationInfo)
};


class LocationSettingsPrivate;
class Q_DECL_EXPORT LocationSettings : public QObject
{
    Q_OBJECT
    Q_PROPERTY(bool gpsAvailable READ gpsAvailable CONSTANT)
    Q_PROPERTY(bool locationEnabled READ locationEnabled WRITE setLocationEnabled NOTIFY locationEnabledChanged)
    Q_PROPERTY(bool gpsEnabled READ gpsEnabled WRITE setGpsEnabled NOTIFY gpsEnabledChanged)
    Q_PROPERTY(LocationMode locationMode READ locationMode WRITE setLocationMode NOTIFY locationModeChanged)
    Q_PROPERTY(DataSources allowedDataSources READ allowedDataSources WRITE setAllowedDataSources NOTIFY allowedDataSourcesChanged)

public:
    enum Mode {
        AsynchronousMode,
        SynchronousMode
    };

    enum LocationMode {
        HighAccuracyMode,
        BatterySavingMode,
        DeviceOnlyMode,
        CustomMode Q_DECL_DEPRECATED
    };
    Q_ENUM(LocationMode)

    // Must be identical to n-q-p-ss::LocationSettings::DataSource.
    // Values are roughly grouped by type, with space for future expansion.
    enum DataSource {
        NoDataSources                   = 0UL,

        OnlineDataSources               = 1UL << 0,

        DeviceSensorsData               = 1UL << 5,
        BluetoothData                   = 1UL << 10,
        WlanData                        = 1UL << 15,
        CellTowerData                   = 1UL << 20,

        GpsData                         = 1UL << 25,
        GlonassData                     = 1UL << 26,
        BeidouData                      = 1UL << 27,
        GalileoData                     = 1UL << 28,
        QzssData                        = 1UL << 29,
        SbasData                        = 1UL << 30,

        LastDataSource                  = 1UL << 31
    };
    Q_ENUM(DataSource)
    Q_DECLARE_FLAGS(DataSources, DataSource)
    Q_FLAG(DataSources)

    LocationSettings(QObject *parent = Q_NULLPTR);
    LocationSettings(Mode mode, QObject *parent = Q_NULLPTR);

    bool gpsAvailable() const;
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    Q_INVOKABLE bool agpsAvailable(const QString &agpsProvider) const;
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    Q_INVOKABLE QStringList availableAgpsProviders() const;

    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    bool locationEnabled() const;
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    bool gpsEnabled() const;
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    bool gpsFlightMode() const;
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    LocationMode locationMode() const;
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    DataSources allowedDataSources() const;
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    Q_INVOKABLE bool agpsEnabled(const QString &agpsProvider) const;
    Q_DECL_DEPRECATED_X(NOT_AVAILABLE_MSG)
    Q_INVOKABLE bool agpsAgreementAccepted(const QString &agpsProvider) const;

    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    void setLocationEnabled(bool value);
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    void setGpsEnabled(bool value);
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    void setGpsFlightMode(bool value);
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    void setLocationMode(LocationMode locationMode);
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    void setAllowedDataSources(DataSources dataSources);
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    Q_INVOKABLE void setAgpsEnabled(const QString &agpsProvider, bool value);
    Q_DECL_DEPRECATED_X(NOT_AVAILABLE_MSG)
    Q_INVOKABLE void setAgpsAgreementAccepted(const QString &agpsProvider, bool value);

    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    Q_INVOKABLE void allowDataSource(DataSource dataSource);
    Q_DECL_DEPRECATED_X(DEPRECATED_MSG)
    Q_INVOKABLE void disallowDataSource(DataSource dataSource);

signals:
    void locationEnabledChanged();
    void gpsEnabledChanged();
    void gpsFlightModeChanged();
    void locationModeChanged();
    void allowedDataSourcesChanged();

private:
    LocationSettingsPrivate *d_ptr;
    Q_DISABLE_COPY(LocationSettings)
    Q_DECLARE_PRIVATE(LocationSettings)
};

#undef DEPRECATED_MSG
#undef NOT_AVAILABLE_MSG

}
}

Q_DECLARE_OPERATORS_FOR_FLAGS(Sailfish::Mdm::LocationSettings::DataSources)
Q_DECLARE_METATYPE(Sailfish::Mdm::LocationInfo::Position)
Q_DECLARE_TYPEINFO(Sailfish::Mdm::LocationInfo::Position, Q_MOVABLE_TYPE);

#endif // SAILFISH_MDM_LOCATIONSETTINGS_H
