// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR LGPL-3.0-only OR GPL-2.0-only OR GPL-3.0-only

#ifndef QFIXED_P_H
#define QFIXED_P_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists for the convenience
// of other Qt classes.  This header file may change from version to
// version without notice, or even be removed.
//
// We mean it.
//

#include "QtCore/qdebug.h"
#include "QtCore/qpoint.h"
#include "QtCore/qsize.h"

QT_BEGIN_NAMESPACE

struct QFixed {
private:
    Q_DECL_CONSTEXPR QFixed(int val, int) : val(val) {} // 2nd int is just a dummy for disambiguation
public:
    Q_DECL_CONSTEXPR QFixed() : val(0) {}
    Q_DECL_CONSTEXPR QFixed(int i) : val(i * 64) {}
    Q_DECL_CONSTEXPR QFixed(long i) : val(i * 64) {}
    QFixed &operator=(int i) { val = i * 64; return *this; }
    QFixed &operator=(long i) { val = i * 64; return *this; }

    Q_DECL_CONSTEXPR static QFixed fromReal(qreal r) { return fromFixed((int)(r*qreal(64))); }
    Q_DECL_CONSTEXPR static QFixed fromFixed(int fixed) { return QFixed(fixed,0); } // uses private ctor

    Q_DECL_CONSTEXPR inline int value() const { return val; }
    inline void setValue(int value) { val = value; }

    Q_DECL_CONSTEXPR inline int toInt() const { return (((val)+32) & -64)>>6; }
    Q_DECL_CONSTEXPR inline qreal toReal() const { return ((qreal)val)/(qreal)64; }

    Q_DECL_CONSTEXPR inline int truncate() const { return val>>6; }
    Q_DECL_CONSTEXPR inline QFixed round() const { return fromFixed(((val)+32) & -64); }
    Q_DECL_CONSTEXPR inline QFixed floor() const { return fromFixed((val) & -64); }
    Q_DECL_CONSTEXPR inline QFixed ceil() const { return fromFixed((val+63) & -64); }

    Q_DECL_CONSTEXPR inline QFixed operator+(int i) const { return fromFixed(val + i * 64); }
    Q_DECL_CONSTEXPR inline QFixed operator+(uint i) const { return fromFixed((val + (i<<6))); }
    Q_DECL_CONSTEXPR inline QFixed operator+(const QFixed &other) const { return fromFixed((val + other.val)); }
    inline QFixed &operator+=(int i) { val += i * 64; return *this; }
    inline QFixed &operator+=(uint i) { val += (i<<6); return *this; }
    inline QFixed &operator+=(const QFixed &other) { val += other.val; return *this; }
    Q_DECL_CONSTEXPR inline QFixed operator-(int i) const { return fromFixed(val - i * 64); }
    Q_DECL_CONSTEXPR inline QFixed operator-(uint i) const { return fromFixed((val - (i<<6))); }
    Q_DECL_CONSTEXPR inline QFixed operator-(const QFixed &other) const { return fromFixed((val - other.val)); }
    inline QFixed &operator-=(int i) { val -= i * 64; return *this; }
    inline QFixed &operator-=(uint i) { val -= (i<<6); return *this; }
    inline QFixed &operator-=(const QFixed &other) { val -= other.val; return *this; }
    Q_DECL_CONSTEXPR inline QFixed operator-() const { return fromFixed(-val); }

    Q_DECL_CONSTEXPR inline bool operator==(const QFixed &other) const { return val == other.val; }
    Q_DECL_CONSTEXPR inline bool operator!=(const QFixed &other) const { return val != other.val; }
    Q_DECL_CONSTEXPR inline bool operator<(const QFixed &other) const { return val < other.val; }
    Q_DECL_CONSTEXPR inline bool operator>(const QFixed &other) const { return val > other.val; }
    Q_DECL_CONSTEXPR inline bool operator<=(const QFixed &other) const { return val <= other.val; }
    Q_DECL_CONSTEXPR inline bool operator>=(const QFixed &other) const { return val >= other.val; }
    Q_DECL_CONSTEXPR inline bool operator!() const { return !val; }

    inline QFixed &operator/=(int x) { val /= x; return *this; }
    inline QFixed &operator/=(const QFixed &o) {
        if (o.val == 0) {
            val = 0x7FFFFFFFL;
        } else {
            bool neg = false;
            qint64 a = val;
            qint64 b = o.val;
            if (a < 0) { a = -a; neg = true; }
            if (b < 0) { b = -b; neg = !neg; }

            int res = (int)(((a << 6) + (b >> 1)) / b);

            val = (neg ? -res : res);
        }
        return *this;
    }
    Q_DECL_CONSTEXPR inline QFixed operator/(int d) const { return fromFixed(val/d); }
    inline QFixed operator/(QFixed b) const { QFixed f = *this; return (f /= b); }
    inline QFixed operator>>(int d) const { QFixed f = *this; f.val >>= d; return f; }
    inline QFixed &operator*=(int i) { val *= i; return *this; }
    inline QFixed &operator*=(uint i) { val *= i; return *this; }
    inline QFixed &operator*=(const QFixed &o) {
        bool neg = false;
        qint64 a = val;
        qint64 b = o.val;
        if (a < 0) { a = -a; neg = true; }
        if (b < 0) { b = -b; neg = !neg; }

        int res = (int)((a * b + 0x20L) >> 6);
        val = neg ? -res : res;
        return *this;
    }
    Q_DECL_CONSTEXPR inline QFixed operator*(int i) const { return fromFixed(val * i); }
    Q_DECL_CONSTEXPR inline QFixed operator*(uint i) const { return fromFixed(val * i); }
    inline QFixed operator*(const QFixed &o) const { QFixed f = *this; return (f *= o); }

private:
    Q_DECL_CONSTEXPR QFixed(qreal i) : val((int)(i*qreal(64))) {}
    QFixed &operator=(qreal i) { val = (int)(i*qreal(64)); return *this; }
    Q_DECL_CONSTEXPR inline QFixed operator+(qreal i) const { return fromFixed((val + (int)(i*qreal(64)))); }
    inline QFixed &operator+=(qreal i) { val += (int)(i*64); return *this; }
    Q_DECL_CONSTEXPR inline QFixed operator-(qreal i) const { return fromFixed((val - (int)(i*qreal(64)))); }
    inline QFixed &operator-=(qreal i) { val -= (int)(i*64); return *this; }
    inline QFixed &operator/=(qreal r) { val = (int)(val/r); return *this; }
    Q_DECL_CONSTEXPR inline QFixed operator/(qreal d) const { return fromFixed((int)(val/d)); }
    inline QFixed &operator*=(qreal d) { val = (int) (val*d); return *this; }
    Q_DECL_CONSTEXPR inline QFixed operator*(qreal d) const { return fromFixed((int) (val*d)); }
    int val;
};
Q_DECLARE_TYPEINFO(QFixed, Q_PRIMITIVE_TYPE);

#define QFIXED_MAX (INT_MAX/256)

Q_DECL_CONSTEXPR inline int qRound(const QFixed &f) { return f.toInt(); }
Q_DECL_CONSTEXPR inline int qFloor(const QFixed &f) { return f.floor().truncate(); }

Q_DECL_CONSTEXPR inline QFixed operator*(int i, const QFixed &d) { return d*i; }
Q_DECL_CONSTEXPR inline QFixed operator+(int i, const QFixed &d) { return d+i; }
Q_DECL_CONSTEXPR inline QFixed operator-(int i, const QFixed &d) { return -(d-i); }
Q_DECL_CONSTEXPR inline QFixed operator*(uint i, const QFixed &d) { return d*i; }
Q_DECL_CONSTEXPR inline QFixed operator+(uint i, const QFixed &d) { return d+i; }
Q_DECL_CONSTEXPR inline QFixed operator-(uint i, const QFixed &d) { return -(d-i); }
// Q_DECL_CONSTEXPR inline QFixed operator*(qreal d, const QFixed &d2) { return d2*d; }

Q_DECL_CONSTEXPR inline bool operator==(const QFixed &f, int i) { return f.value() == i * 64; }
Q_DECL_CONSTEXPR inline bool operator==(int i, const QFixed &f) { return f.value() == i * 64; }
Q_DECL_CONSTEXPR inline bool operator!=(const QFixed &f, int i) { return f.value() != i * 64; }
Q_DECL_CONSTEXPR inline bool operator!=(int i, const QFixed &f) { return f.value() != i * 64; }
Q_DECL_CONSTEXPR inline bool operator<=(const QFixed &f, int i) { return f.value() <= i * 64; }
Q_DECL_CONSTEXPR inline bool operator<=(int i, const QFixed &f) { return i * 64 <= f.value(); }
Q_DECL_CONSTEXPR inline bool operator>=(const QFixed &f, int i) { return f.value() >= i * 64; }
Q_DECL_CONSTEXPR inline bool operator>=(int i, const QFixed &f) { return i * 64 >= f.value(); }
Q_DECL_CONSTEXPR inline bool operator<(const QFixed &f, int i) { return f.value() < i * 64; }
Q_DECL_CONSTEXPR inline bool operator<(int i, const QFixed &f) { return i * 64 < f.value(); }
Q_DECL_CONSTEXPR inline bool operator>(const QFixed &f, int i) { return f.value() > i * 64; }
Q_DECL_CONSTEXPR inline bool operator>(int i, const QFixed &f) { return i * 64 > f.value(); }

/*Implementation copied from KDE's Qt 5.15 branch:
 * https://invent.kde.org/qt/qt/qtbase/-/commit/2b8674a2fc67089ff87d3751ef2977cbb37616a2
 */
template <typename T> inline typename std::enable_if<std::is_signed<T>::value, bool>::type
add_overflow(T v1, T v2, T *r)
{
    // Here's how we calculate the overflow:
    // 1) unsigned addition is well-defined, so we can always execute it
    // 2) conversion from unsigned back to signed is implementation-
    //    defined and in the implementations we use, it's a no-op.
    // 3) signed integer overflow happens if the sign of the two input operands
    //    is the same but the sign of the result is different. In other words,
    //    the sign of the result must be the same as the sign of either
    //    operand.

    using U = typename std::make_unsigned<T>::type;
    *r = T(U(v1) + U(v2));

    // If int is two's complement, assume all integer types are too.
    if (std::is_same<int32_t, int>::value) {
        // Two's complement equivalent (generates slightly shorter code):
        //  x ^ y             is negative if x and y have different signs
        //  x & y             is negative if x and y are negative
        // (x ^ z) & (y ^ z)  is negative if x and z have different signs
        //                    AND y and z have different signs
        return ((v1 ^ *r) & (v2 ^ *r)) < 0;
    }

    bool s1 = (v1 < 0);
    bool s2 = (v2 < 0);
    bool sr = (*r < 0);
    return s1 != sr && s2 != sr;
    // also: return s1 == s2 && s1 != sr;
}

inline bool qAddOverflow(QFixed v1, QFixed v2, QFixed *r)
{
    int val;
    bool result = add_overflow(v1.value(), v2.value(), &val);
    r->setValue(val);
    return result;
}

#ifndef QT_NO_DEBUG_STREAM
inline QDebug &operator<<(QDebug &dbg, const QFixed &f)
{ return dbg << f.toReal(); }
#endif

struct QFixedPoint {
    QFixed x;
    QFixed y;
    Q_DECL_CONSTEXPR inline QFixedPoint() {}
    Q_DECL_CONSTEXPR inline QFixedPoint(const QFixed &_x, const QFixed &_y) : x(_x), y(_y) {}
    Q_DECL_CONSTEXPR QPointF toPointF() const { return QPointF(x.toReal(), y.toReal()); }
    Q_DECL_CONSTEXPR static QFixedPoint fromPointF(const QPointF &p) {
        return QFixedPoint(QFixed::fromReal(p.x()), QFixed::fromReal(p.y()));
    }
};
Q_DECLARE_TYPEINFO(QFixedPoint, Q_PRIMITIVE_TYPE);

Q_DECL_CONSTEXPR inline QFixedPoint operator-(const QFixedPoint &p1, const QFixedPoint &p2)
{ return QFixedPoint(p1.x - p2.x, p1.y - p2.y); }
Q_DECL_CONSTEXPR inline QFixedPoint operator+(const QFixedPoint &p1, const QFixedPoint &p2)
{ return QFixedPoint(p1.x + p2.x, p1.y + p2.y); }

struct QFixedSize {
    QFixed width;
    QFixed height;
    Q_DECL_CONSTEXPR QFixedSize() {}
    Q_DECL_CONSTEXPR QFixedSize(QFixed _width, QFixed _height) : width(_width), height(_height) {}
    Q_DECL_CONSTEXPR QSizeF toSizeF() const { return QSizeF(width.toReal(), height.toReal()); }
    Q_DECL_CONSTEXPR static QFixedSize fromSizeF(const QSizeF &s) {
        return QFixedSize(QFixed::fromReal(s.width()), QFixed::fromReal(s.height()));
    }
};
Q_DECLARE_TYPEINFO(QFixedSize, Q_PRIMITIVE_TYPE);

QT_END_NAMESPACE

#endif // QTEXTENGINE_P_H
