/****************************************************************************************
**
** SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: Proprietary
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0

SilicaMouseArea {
    id: searchField

    property real preferredWidth
    property alias placeholderText: textField.placeholderText
    property alias text: textField.text
    property bool backgroundEnabled
    readonly property bool __fillWidth: !preferredWidth
    property bool __auroracontrols_appbarSearch
    readonly property real minimalWidth: Theme.dp(250)
    property string context
    property bool _highlightedByAppBar

    onClicked: textField.forceActiveFocus()
    onVisibleChanged: if (parent) parent._updateComponentsWidth()
    onPreferredWidthChanged: if (parent) parent._updateComponentsWidth()
    on__FillWidthChanged: if (parent) parent._updateComponentsWidth()
    onActiveFocusChanged: if (activeFocus) textField.forceActiveFocus()

    onEntered: {
        if (parent && context && !textField.highlighted) {
            if (parent._contextVisible) {
                if (parent && context) parent._setContext(searchField.context)
            } else {
                hoverTimer.start()
            }
        }
    }
    onExited: {
        parent._resetContextNow()
        hoverTimer.stop()
    }
    onPressed: {
        parent._resetContextNow()
        hoverTimer.stop()
    }

    height: Theme.dp(72)
    width: Math.max(preferredWidth ? preferredWidth : parent._fillComponentsWidth, minimalWidth)
    cursorShape: enabled ? Qt.IBeamCursor : Qt.ArrowCursor

    hoverEnabled: true

    Timer {
        id: hoverTimer
        interval: 1000
        onTriggered: if (searchField.parent && searchField.context) searchField.parent._setContext(searchField.context)
    }

    Rectangle {
        anchors.fill: parent

        visible: backgroundEnabled
        radius: Theme.dp(6)
        color: textField.highlighted ? palette.highlightColor : palette.primaryColor
        opacity: 0.1
    }

    SearchField {
        id: textField

        anchors.fill: parent

        highlighted: searchField.highlighted || textField.activeFocus || searchField._highlightedByAppBar
        font.pixelSize: Theme.fontSizeMedium
        textLeftMargin: Theme.iconSizeMedium + Theme.paddingMedium
        textRightMargin: 0

        leftItem: Icon {
            source: "image://theme/icon-splus-search"
        }

        rightItem: IconButton {
            onClicked: textField.text = ""

            width: Theme.iconSizeMedium
            height: Theme.iconSizeMedium
            enabled: textField.text.length > 0
            opacity: enabled  ? 1.0 : 0.0
            icon.source: "image://theme/icon-m-clear"
            icon.sourceSize: Qt.size(Theme.iconSizeSmallPlus, Theme.iconSizeSmallPlus)

            Behavior on opacity { FadeAnimation {} }
        }

        onHighlightedChanged: {
            if (highlighted) {
                searchField.parent._resetContextNow()
                hoverTimer.stop()
            }
        }
    }

    Component.onCompleted: {
        if (preferredWidth != 0 && preferredWidth < minimalWidth) {
            console.error("AppBarSearchField width can't be lower than %1 pixels.".arg(minimalWidth))
        }
    }
}
