/*
 * SPDX-FileCopyrightText: 2013–2019 Jolla Ltd
 * SPDX-FileCopyrightText: 2019-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef DEVELOPERMODESETTINGS_H
#define DEVELOPERMODESETTINGS_H

#include <QObject>

#include <systemsettingsglobal.h>
#include <daemon.h>

#include <nemo-dbus/connection.h>
#include <nemo-dbus/interface.h>

class MGConfItem;

QT_BEGIN_NAMESPACE
class QDBusPendingCallWatcher;
QT_END_NAMESPACE

class SYSTEMSETTINGS_EXPORT DeveloperModeSettings : public QObject
{
    Q_OBJECT
    Q_ENUMS(Status)
    Q_ENUMS(Error)

    Q_PROPERTY(QString wlanIpAddress READ wlanIpAddress NOTIFY wlanIpAddressChanged)
    Q_PROPERTY(QString usbIpAddress READ usbIpAddress NOTIFY usbIpAddressChanged)
    Q_PROPERTY(QString username READ username CONSTANT)
    Q_PROPERTY(enum DeveloperModeSettings::Error error READ error NOTIFY errorChanged)
    Q_PROPERTY(bool developerModeEnabled READ developerModeEnabled NOTIFY developerModeEnabledChanged)
    Q_PROPERTY(bool remoteLoginEnabled READ remoteLoginEnabled NOTIFY remoteLoginEnabledChanged)
    Q_PROPERTY(bool validationEnabled READ validationEnabled NOTIFY validationEnabledChanged)
    Q_PROPERTY(bool silentInstallationEnabled READ silentInstallationEnabled NOTIFY silentInstallationEnabledChanged)
    Q_PROPERTY(enum DeveloperModeSettings::Status workStatus READ workStatus NOTIFY workStatusChanged)
    Q_PROPERTY(bool repositoryAccessRequired READ repositoryAccessRequired NOTIFY repositoryAccessRequiredChanged)
    Q_PROPERTY(bool debugHomeEnabled READ debugHomeEnabled NOTIFY debugHomeEnabledChanged)
    Q_PROPERTY(bool autoTestUIEnabled READ autoTestUIEnabled NOTIFY autoTestUIEnabledChanged)
    Q_PROPERTY(bool isEmulatorDevice READ isEmulatorDevice CONSTANT)

public:
    explicit DeveloperModeSettings(QObject *parent = NULL);
    virtual ~DeveloperModeSettings();

    enum Status {
        Idle = 0,
        InstallingPackages,
        EnablingHomeDebug,
    };

    // TODO: provide dev-mode plugin as part of developer mode package
    // to avoid unnecessary code dupliction and ease maintanence
    enum Error {
        NoError,
        DisablingDeveloperMode,
        ServiceBlocked,
        DeveloperModeDisabled,
        Access,
        InstallingDeveloperModePackages,
        AllowConnections,
        RemoteLogin,
        MovingDebugSymbols,
        NotEnoughSpace,
        AutoTestUIService,
        PermissionFile,
        // This one should be last as it is not part of developer mode daemon errors
        DbusFail,
    };

    QString wlanIpAddress() const;
    QString usbIpAddress() const;
    QString username() const;
    enum DeveloperModeSettings::Error error() const;
    bool developerModeEnabled() const;
    bool remoteLoginEnabled() const;
    bool validationEnabled() const;
    bool silentInstallationEnabled() const;
    enum DeveloperModeSettings::Status workStatus() const;
    bool repositoryAccessRequired() const;
    bool debugHomeEnabled() const;
    bool autoTestUIEnabled() const;
    bool isEmulatorDevice() const;

    Q_INVOKABLE void setDeveloperMode(bool enabled);
    Q_INVOKABLE void setRemoteLoginEnabled(bool enabled);
    Q_INVOKABLE void setValidationEnabled(bool enabled);
    Q_INVOKABLE void setSilentInstallationEnabled(bool enabled);
    Q_INVOKABLE void setUsbIpAddress(const QString &usbIpAddress);
    Q_INVOKABLE void refresh();
    Q_INVOKABLE void moveDebugToHome(bool enabled, bool force);
    Q_INVOKABLE void setAutoTestUIEnabled(bool enabled);

signals:
    void wlanIpAddressChanged();
    void usbIpAddressChanged();
    void developerModeEnabledChanged();
    void remoteLoginEnabledChanged();
    void validationEnabledChanged();
    void silentInstallationEnabledChanged();
    void workStatusChanged();
    void repositoryAccessRequiredChanged();
    void debugHomeEnabledChanged();
    void autoTestUIEnabledChanged();
    void errorChanged();

private slots:
    void newDeveloperModeValue(bool enabled, bool isSignaled = true);
    void newRemoteLoginValue(bool enabled);
    void newValidationValue(bool enabled);
    void newSilentInstallationValue(bool enabled);
    void newHomeDebugValue(bool enabled);
    void newAutoTestUIValue(bool enabled);

private:
    void resetState();
    void setWorkStatus(Status status);

    void loadDeveloperModeValue();
    void loadRemoteLoginValue();
    void loadValidationValue();
    void loadSilentInstallationValue();
    void loadHomeDebugValue();
    void loadAutoTestUIValue();

    QString usbModedGetConfig(const QString &key, const QString &fallback);
    void usbModedSetConfig(const QString &key, const QString &value);

    void checkError(const QDBusError &error);
    void retrieveError();

    NemoDBus::Connection m_connection;
    NemoDBus::Interface m_usbModeDaemon;
    NemoDBus::Interface m_developerModeDaemon;
    QString m_wlanIpAddress;
    QString m_usbInterface;
    QString m_usbIpAddress;
    QString m_username;
    DeveloperModeSettings::Error m_lastError;
    bool m_developerModeEnabled;
    bool m_remoteLoginEnabled;
    bool m_validationEnabled;
    bool m_silentInstallationEnabled;
    bool m_debugHomeEnabled;
    bool m_autoTestUIEnabled;
    bool m_isEmulatorDevice;
    DeveloperModeSettings::Status m_workStatus;
};

Q_DECLARE_METATYPE(DeveloperModeSettings::Status)

#endif /* DEVELOPERMODESETTINGS_H */
