/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013 Jolla Ltd
** SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.2
import Sailfish.Silica 1.0

SilicaItem {
    id: root

    property bool running
    property int size: BusyIndicatorSize.Medium
    property alias color: busyIndicator.color
    property bool _forceAnimation

    implicitWidth: busyIndicator.width
    implicitHeight: busyIndicator.height
    visible: opacity > 0
    opacity: running ? 1.0 : 0.0

    Behavior on opacity { FadeAnimation {}}

    ShaderEffect {
        id: busyIndicator

        readonly property real antialias: 4 / width
        property color color: palette.highlightColor
        property real time

        anchors.centerIn: parent

        width: switch (root.size) {
            case BusyIndicatorSize.ExtraSmall:
                return Theme.iconSizeExtraSmall
            case BusyIndicatorSize.Small:
                return Theme.iconSizeSmall
            case BusyIndicatorSize.Medium:
                return Theme.iconSizeMedium
            case BusyIndicatorSize.Launcher:
                return Theme.iconSizeLauncher
            default: // BusyIndicatorSize.Large
                return Theme.iconSizeExtraLarge
        }
        height: width

        vertexShader: "
#define COUNT      10 // Should be equal to fragmentShader.COUNT and busyIndicator.count
#define PI2        1.57079632679 // pi / 2
#define TAU        6.28318530718 // pi * 2
#define START_SIZE 0.09 // default start blob size

precision lowp float;

uniform highp mat4 qt_Matrix;
attribute highp vec4 qt_Vertex;
attribute highp vec2 qt_MultiTexCoord0;

varying highp vec2 coord;
varying vec2 pointArray[COUNT];
varying float sizeArray[COUNT];
varying float opacityArray[COUNT];

uniform float time;

float easeInOutQuad(float x) {
    return x < 0.5 ? 2.0 * x * x : 1.0 - pow(-2.0 * x + 2.0, 2.0) * 0.5;
}

float easeInOutQuart(float x) {
    return x < 0.5 ? 8.0 * x * x * x * x : 1.0 - pow(-2.0 * x + 2.0, 4.0) * 0.5;
}

void main() {
    coord = 2.0 * qt_MultiTexCoord0 - 1.0;

    const float count = float(COUNT);
    const float delta = 0.14;
    const float delta2 = delta * 0.5;
    const float delta4 = delta2 * 0.5;
    const float delta8 = delta4 * 0.5;

    float counter = floor(time);
    float ct = time - 1. * counter;
    float dt = ct * delta * count * 0.25 - delta4;
    float rotation = (360.0 / count * TAU / 180.0) * (easeInOutQuart(ct) + counter) - PI2;
    float scale = (0.625 + 0.375 * (0.5 - abs(0.5 - easeInOutQuad(ct))));

    for (int i = 0; i < COUNT; i++) {
        float fi = float(i);
        float angle = fi / count * TAU + rotation;
        float size = delta2 - abs(delta2 - min(delta, max(0.0, dt - fi * delta8)));
        pointArray[i] = vec2(cos(angle), sin(angle)) * scale;
        sizeArray[i] = size + START_SIZE;
        opacityArray[i] = 0.4 + 0.6 * (2.0 * size / delta);
    }

    gl_Position = qt_Matrix * qt_Vertex;
}
        "

        fragmentShader: "
#define COUNT 10 // Should be equal to vertexShader.COUNT and busyIndicator.count

precision lowp float;

uniform vec4 color;
uniform float antialias;
uniform float qt_Opacity;

varying highp vec2 coord;
varying vec2 pointArray[COUNT];
varying float sizeArray[COUNT];
varying float opacityArray[COUNT];

void main() {
    float value = 0.0;
    for (int i = 0; i < COUNT; i++) {
        float blob = 1.0 - clamp((length(coord - pointArray[i]) - sizeArray[i]) / antialias, 0.0, 1.0);
        value += blob * opacityArray[i];
    }
    gl_FragColor = color * value * qt_Opacity;
}
        "

        UniformAnimator {
            target: busyIndicator
            uniform: "time"
            from: 0; to: 5
            duration: 5000
            running: root.running || (root.opacity > 0 && root.visible)
            loops: Animation.Infinite
        }
    }
}
