/*
 * SPDX-FileCopyrightText: 2018-2019 Jolla Ltd.
 * SPDX-FileCopyrightText: 2019-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import QtGraphicalEffects 1.0
import Sailfish.Silica 1.0
import Nemo.Configuration 1.0

ListItem {
    id: root

    property string fileName
    property string mimeType
    property string fileLocation
    property double size
    property bool isDir
    property var created
    property var modified
    property bool compressed
    property int nestedFiles

    readonly property alias icon: icon
    property string searchText
    property alias textFormat: nameLabel.textFormat

    property bool showStorageIcon

    readonly property var internalStorageRegExp: new RegExp("^" + StandardPaths.home)
    readonly property bool isInternalStorage: internalStorageRegExp.test(fileLocation)
    readonly property bool isContextMenu: _menuItem != null && _menuItem.hasOwnProperty("_expanded")

    function handleMouseRightClick(mouse) {
        if (mouse.button !== Qt.RightButton)
            return false

        root.pressAndHold(mouse)
        return true
    }

    contentWidth: isContextMenu ? parent.width : parent.width - Theme.paddingMedium
    contentHeight: Theme.iconSizeLauncher + Theme.paddingSmall * 3
    contentItem.radius: isContextMenu ? 0 : Theme.paddingSmall
    contentItem.anchors.centerIn: isContextMenu ? undefined : root
    _backgroundColor: Theme.rgba(palette.highlightBackgroundColor, _showPress || menuOpen ? Theme.highlightBackgroundOpacity : 0)
    acceptedButtons: Qt.LeftButton | Qt.RightButton

    ConfigurationValue {
        id: timeFormatConfig

        key: "/sailfish/i18n/lc_timeformat24h"
    }

    Row {
        anchors.centerIn:  root.contentItem
        height: Theme.iconSizeLauncher
        spacing: Theme.paddingMedium

        Item {
            width: Theme.iconSizeLauncher
            height: width

            Rectangle {
                id: mask

                anchors.fill: parent
                radius: Theme.paddingSmall
                visible: false
            }

            Rectangle {
                id: imageBackground

                anchors.fill: parent
                visible: false

                gradient: Gradient {
                    // Abusing gradient for inactive mimeTypes
                    GradientStop { position: 0.0; color: Theme.rgba(Theme.primaryColor, 0.1) }
                    GradientStop { position: 1.0; color: "transparent" }
                }

                HighlightImage {
                    id: icon
                    objectName: "FileIconHighlightImage"

                    anchors.centerIn: parent
                    source: root.isDir
                            ? "image://theme/icon-m-file-folder"
                            : Theme.iconForMimeType(root.mimeType)
                }

                Image {
                    objectName: "FileImageHighlightImage"

                    anchors {
                        top: parent.top
                        right: parent.right
                    }
                    visible: compressed

                    source: {
                        var iconSource = "image://theme/icon-s-file-compressed-displaced"
                        return iconSource + (highlighted ? "?" + Theme.highlightColor : "")
                    }
                }
            }

            OpacityMask {
                anchors.fill: parent
                source: imageBackground
                maskSource: mask
            }
        }

        Column {
            width: root.width - root.contentHeight - parent.spacing - Theme.horizontalPageMargin + Theme.paddingMedium
            anchors.verticalCenter: parent.verticalCenter
            spacing: Theme.paddingSmall

            Label {
                id: nameLabel
                objectName: "FileItemNameLabel"

                text: {
                    var _fileName = root.fileName.indexOf("/") >= 0
                            ? root.fileName.substring(root.fileName.lastIndexOf("/") + 1, root.fileName.length)
                            : root.fileName

                    return root.searchText.length > 0 ? Theme.highlightText(_fileName, root.searchText.replace("*", ""), Theme.highlightColor)
                                                      : _fileName
                }
                width: parent.width
                truncationMode: TruncationMode.Fade
            }

            Label {
                id: sizeLabel
                objectName: "SizeLabel"

                width: nameLabel.width
                text: root.isDir
                      //% "%n object(s)"
                      ? qsTrId("filemanager-la-nested_objects", root.nestedFiles)
                      : Format.formatFileSize(root.size)

                color: highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                horizontalAlignment: Qt.AlignRight
                font.pixelSize: Theme.fontSizeExtraSmall

                Label {
                    id: dateLabel
                    objectName: "DateLabel"

                    width: parent.width
                    text: Qt.formatDateTime(root.modified || root.created, "d MMMM yyyy hh:mm" + (timeFormatConfig.value === "12" ? " AP" : ""))
                    color: highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                    horizontalAlignment: Qt.AlignLeft
                    font.pixelSize: Theme.fontSizeExtraSmall
                }

                Loader {
                    id: storageIcon

                    readonly property int rightMargin: active ? Theme.horizontalPageMargin : 0

                    anchors {
                        left: dateLabel.right
                        verticalCenter: dateLabel.verticalCenter
                    }

                    sourceComponent: storageIconComponent
                    active: showStorageIcon
                }
            }
        }

        Component {
            id: storageIconComponent

            StorageIcon {
                isInternal: root.isInternalStorage
            }
        }
    }
}
