/*
 * SPDX-FileCopyrightText: 2017-2020 Jolla Ltd
 * SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef SAILFISH_MDM_CERTIFICATE_H
#define SAILFISH_MDM_CERTIFICATE_H

#include <QObject>
#include <QSharedDataPointer>
#include <QHash>

QT_BEGIN_NAMESPACE
class QSslCertificate;
QT_END_NAMESPACE

namespace Sailfish {
namespace Mdm {

class Certificates;
class CertificateFilePrivate;
class Q_DECL_EXPORT CertificateFile
{
public:
    CertificateFile();
    CertificateFile(const CertificateFile &file);
    ~CertificateFile();

    CertificateFile &operator=(const CertificateFile &file);

    bool isNull() const;

    QString fileName() const;
    QList<QSslCertificate> certificates() const;

private:
    friend class Certificates;
    friend class CertificatesPrivate;

    CertificateFile(const QString &name, const QList<QSslCertificate> &certs);

    QExplicitlySharedDataPointer<CertificateFilePrivate> d;
};

class CertificatesPrivate;
class Q_DECL_EXPORT Certificates : public QObject
{
    Q_OBJECT
public:
    enum Result {
        Success,
        Error
    };

    enum ErrorType : int {
        NoError,
        InvalidCommand,
        InvalidFileName,
        EmptyFile,
        FileAlreadyExist,
        CantOpenFile,
        CantWriteFile,
        FailedVerify,
        FailedFinalizeFile,
        UpdateCaTrusIssue,
        DBusError,
        ErrorDecode,
        ErrorPassphrase,
        InvalidId,
        NoActiveUser,
    };

    struct UserCertInfo {
        QString path;
        QString password;
    };

    explicit Certificates(QObject *parent = nullptr);
    ~Certificates();

    QList<CertificateFile> installedFiles() const;
    QList<CertificateFile> blacklistedFiles() const;
    QList<QSslCertificate> certificates() const;
    QHash<QString, QSslCertificate> userCertificates() const;

    void install(const QString &fileName, const QByteArray &data);
    void install(const QString &fileName, const QList<QSslCertificate> &certificates);
    void uninstall(const QString &fileName);

    void userInstall(const QList<UserCertInfo> &certs);
    void userUninstall(const QStringList &ids);

    void blacklist(const QString &fileName, const QByteArray &data);
    void blacklist(const QString &fileName, const QList<QSslCertificate> &certificates);
    void unblacklist(const QString &fileName);
    int lastErrorType();
    QString lastErrorMessage();

signals:
    void installFinished(const CertificateFile &file, Result result);
    void uninstallFinished(const CertificateFile &file, Result result);
    void blacklistFinished(const CertificateFile &file, Result result);
    void unblacklistFinished(const CertificateFile &file, Result result);
    void userInstallFinished(const QString &path, Result result);
    void userUninstallFinished(const QString &id, Result result);

    void certificatesChanged();
    void installedFilesChanged();
    void blacklistedFilesChanged();

private:
    QScopedPointer<CertificatesPrivate> d;

    Q_DISABLE_COPY(Certificates)
};

} // namespace Mdm
} // namespace Sailfish

#endif // SAILFISH_MDM_CERTIFICATE_H
