/*
 * SPDX-FileCopyrightText: Copyright 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include "idownloadhandler.h"

class QQmlEngine;
class QJSEngine;

namespace Aurora {
namespace WebView {

/**
 * @class DownloadHandler
 * @brief Use this singleton to handle file downloads
 * 
 * @link handlers.qml Usage Example @endlink
 */
class DownloadHandler : public QObject, public IDownloadHandler
{
    Q_OBJECT
    /// set to true to allow download requests
    ///
    /// true by default
    Q_PROPERTY(
        bool downloadRequestsAllowed READ downloadRequestsAllowed WRITE setDownloadRequestsAllowed)
public:
    /// @brief default destructor
    ~DownloadHandler() = default;

    /// @brief creates (if doesn't exist) and returns an instance of DownloadHandler
    /// @return DownloadHandler instance
    static QSharedPointer<DownloadHandler> GetInstance(
        QQmlEngine *qmlEngine = nullptr, QJSEngine *jsEngine = nullptr);

    bool downloadRequestsAllowed() override;
    void setDownloadRequestsAllowed(bool allowed) override;
    void notifyDownloadRequested(const DownloadItem &downloadItem, QString suggestedName) override;
    void notifyDownloadUpdated(const DownloadItem &downloadItem) override;

signals:
    /// emitted when a new download is requested
    void downloadRequested(DownloadItem downloadItem, QString suggestedName);
    /// emitted when an existing download status is updated (e.g., on progress change)
    void downloadUpdated(DownloadItem downloadItem);
    /// emitted when a download is completed
    void downloadCompleted(DownloadItem downloadItem);
    /// emitted when a download is interrupted
    void downloadInterrupted(DownloadItem downloadItem);

private:
    explicit DownloadHandler();

    bool m_downloadRequestsAllowed;
};

} // namespace WebView
} // namespace Aurora
