/****************************************************************************************
**
** SPDX-FileCopyrightText: 2013-2019 Jolla Ltd.
** SPDX-FileCopyrightText: 2022-2025 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.6
import QtQuick.Window 2.3 as QtQuick
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import "private"
import "private/Util.js" as Util

BackgroundItem {
    id: dialogHeader

    property Item dialog
    property Flickable flickable
    property string acceptText: defaultAcceptText
    property string cancelText: defaultCancelText
    property alias title: titleText.text
    property bool acceptTextVisible: acceptText.length > 0 && acceptLabel.visible
    property alias extraContent: extraContentPlaceholder
    property bool reserveExtraContent: extraContentPlaceholder.children.length > 0
    property real spacing: Theme.paddingLarge
    property real leftMargin: Theme.horizontalPageMargin
    property real rightMargin: Theme.horizontalPageMargin

    property bool background
    property bool monochrome
    property bool divider: true

    default property alias _children: acceptButton.data
    property int _depth: dialog && dialog._depth ? dialog._depth+2 : 1
    property bool _glassOnly

    property bool _navigatingBack: dialog && dialog._navigationPending === PageNavigation.Back
    property bool _navigatingForward: dialog && dialog._navigationPending === PageNavigation.Forward
    property bool _canGoBack: dialog && dialog.backNavigation && dialog._depth !== 0
    property bool _backgroundVisible: !__silica_applicationwindow_instance._rotating
    property real _maxButtonSize: dialog.width - Theme.itemSizeLarge
    property real _overlayHeight: overlay.height

    //% "Accept"
    property string defaultAcceptText: qsTrId("components-he-dialog_accept")

    //% "Cancel"
    property string defaultCancelText: qsTrId("components-he-dialog_cancel")

    readonly property int _contentOrientation: QtQuick.Window.window
                                               ? QtQuick.Window.window.contentOrientation
                                               : QtQuick.Screen.orientation

    // TODO: Remove top-level BackgroundItem, now here for API compatibility
    down: false
    highlighted: false
    highlightedColor: "transparent"

    height: overlay.height + overlay.topInset + (title.length > 0 ? titleText.height + Theme.paddingMedium : 0) + spacing
    width: parent ? parent.width : Screen.width

    onFlickableChanged: {
        if (flickable) {
            overlay.parent = flickable.contentItem
        } else {
            overlay.parent = dialogHeader
        }
    }

    onBackgroundChanged: if (overlay.active) _setStatusBar()

    Component.onDestruction: Private.StatusBarWatcher.restoreStatusBar(false, overlay)

    Component.onCompleted: {
        if (!dialog)
            dialog = _findDialog()
        if (dialog) {
            dialog._dialogHeader = dialogHeader
        } else {
            console.log("DialogHeader must have a parent Dialog instance")
        }
        if (!flickable)
            flickable = Util.findFlickable(dialogHeader)

        if (Private.StatusBarWatcher.window != __silica_applicationwindow_instance) {
            Private.StatusBarWatcher.window = __silica_applicationwindow_instance
        }
    }

    function _findDialog() {
        var r = parent
        while (r && !r.hasOwnProperty('__silica_dialog'))
            r = r.parent
        return r
    }

    Item {
        id: overlay

        readonly property bool active: dialog && dialog.status === PageStatus.Active

        readonly property real topInset: {
            const topMargin = SafeZoneRect.insets.top
            if (dialogHeader._contentOrientation & Orientation.LandscapeMask
                    || __silica_applicationwindow_instance.displayMode === ApplicationDisplayMode.SafeZone) {
                return topMargin
            }

            const threshold = Theme.dp(20)

            if (dialogHeader._contentOrientation & Orientation.Portrait) {
                if (Private.ScreenCorners.topLeft > threshold || Private.ScreenCorners.topRight > threshold) {
                    return Math.max(topMargin, Private.ScreenCorners.topLeft * 0.5, Private.ScreenCorners.topRight * 0.5)
                }
                return topMargin
            }

            if (Private.ScreenCorners.bottomLeft > threshold || Private.ScreenCorners.bottomRight > threshold) {
                return Math.max(topMargin, Private.ScreenCorners.bottomLeft * 0.5, Private.ScreenCorners.bottomRight * 0.5)
            }
            return topMargin
        }


        readonly property real leftCutoutInset: {
            if (__silica_applicationwindow_instance.displayMode !== ApplicationDisplayMode.FillScreen
                    || !(dialogHeader._contentOrientation & Orientation.LandscapeMask)
                    || (dialogHeader._contentOrientation & Orientation.LandscapeMask && dialogHeader.width < Screen.height)) {
                return 0
            }

            const cutOutRect = Private.CutoutArea.cutoutRect

            if (cutOutRect.y < Screen.height * 0.5 && (dialogHeader._contentOrientation & Orientation.Landscape)) {
                if (cutOutRect.x + cutOutRect.width > Screen.width - Theme._itemSizeHeader) {
                    return cutOutRect.y + cutOutRect.height
                }
            } else if (cutOutRect.y >= Screen.height * 0.5 && (dialogHeader._contentOrientation & Orientation.LandscapeInverted)) {
                if (cutOutRect.x < Theme._itemSizeHeader) {
                    return Screen.height - cutOutRect.y + cutOutRect.height
                }
            }

            return 0
        }

        readonly property real rightCutoutInset: {
            if (__silica_applicationwindow_instance.displayMode !== ApplicationDisplayMode.FillScreen
                    || !(dialogHeader._contentOrientation & Orientation.LandscapeMask)
                    || (dialogHeader._contentOrientation & Orientation.LandscapeMask && dialogHeader.width < Screen.height)) {
                return 0
            }

            const cutOutRect = Private.CutoutArea.cutoutRect

            if (cutOutRect.y < Screen.height * 0.5 && (dialogHeader._contentOrientation & Orientation.LandscapeInverted)) {
                if (cutOutRect.x < Theme._itemSizeHeader) {
                    return cutOutRect.y + cutOutRect.height
                }
            } else if (cutOutRect.y >= Screen.height * 0.5 && (dialogHeader._contentOrientation & Orientation.Landscape)) {
                if (cutOutRect.x + cutOutRect.width > Screen.width - Theme._itemSizeHeader) {
                    return Screen.height - cutOutRect.y + cutOutRect.height
                }
            }
            return 0
        }

        readonly property real maximumAllowedWidth: dialog.width - rightCutoutInset - leftCutoutInset

        function _setStatusBar() {
            if (dialogHeader.background) {
                Private.StatusBarWatcher.setStatusBar(backgroundRect.backgroundColor, 1, overlay)
            } else {
                Private.StatusBarWatcher.setStatusBar(backgroundRect.backgroundColor, 0, overlay)
            }
        }

        onActiveChanged: {
            if (active) {
                return _setStatusBar()
            }

            Private.StatusBarWatcher.restoreStatusBar(false, overlay)
        }

        z: Theme.ZOrderHeader // Just below pulley menu
        height: Theme._itemSizeHeader
        width: dialogHeader.width
        y: (flickable ? Math.max(flickable.contentY, flickable.originY) : 0) + topInset
        x: flickable ? dialogHeader.x : 0
        visible: dialogHeader.visible

        Item {
            anchors.fill: parent
            anchors.topMargin: -parent.topInset

            Private.BackgroundRectangle {
                anchors.fill: parent
                visible: backgroundLoader.status !== Loader.Ready && backgroundRect.opacity != 1
                color: __silica_applicationwindow_instance._backgroundColor
            }

            Loader {
                id: backgroundLoader

                anchors.fill: parent

                sourceComponent: dialogHeader.dialog ? dialogHeader.dialog.background : null
            }

            Rectangle {
                id: backgroundRect

                readonly property color backgroundColor: {
                    if (dialogHeader.monochrome) {
                        return Qt.tint(Theme.rgba(palette.primaryColor, 1.0),
                                       palette.colorScheme === Theme.DarkOnLight ? "#e6ffffff" : "#e6000000") // 90%
                    }
                    return Qt.tint(Theme.rgba(palette.highlightColor, 1.0),
                                   palette.colorScheme === Theme.DarkOnLight ? "#ccffffff" : "#cc000000") // 80%
                }

                onBackgroundColorChanged: if (overlay.active && background) _setStatusBar()

                anchors.fill: parent

                visible: opacity > 0
                opacity: background ? 1 : 0

                color: backgroundColor

                Behavior on opacity {NumberAnimation {duration: 200}}
                Behavior on color {ColorAnimation {duration: 200}}
            }
        }

        FontMetrics {
            id: fontMetrics

            property bool smallFont: boundingRect(cancelLabel.text + acceptLabel.text).width
                                     > (dialogHeader.width - dialogHeader.leftMargin - dialogHeader.rightMargin - Theme.paddingMedium)
            font.pixelSize: Theme.fontSizeLarge
        }

        BackgroundItem {
            id: cancelButton
            property real preferredWidth: Math.min(cancelLabel.implicitWidth*(reserveExtraContent?1.0:cancelLabel.opacity)
                                                        + Theme.paddingLarge + Theme.horizontalPageMargin,
                                                   _maxButtonSize)
            height: overlay.height
            anchors {
                left: parent.left
                leftMargin: overlay.leftCutoutInset
                right: reserveExtraContent ? extraContentPlaceholder.left : acceptButton.left
            }
            enabled: cancelText !== ""
            onClicked: dialog.reject()
            down: false
            highlighted: false
            highlightedColor: "transparent"

            Binding {
                when: overlay.active && _canGoBack
                target: pageStack._pageStackIndicator
                property: "backIndicatorHighlighted"
                value: (pageStack._pageStackIndicator && pageStack._pageStackIndicator.backIndicatorDown) || cancelButton.containsMouse
            }

            Label {
                id: cancelLabel

                objectName: "labelCancelDialog"
                x: dialogHeader.leftMargin
                width: Math.min(overlay.maximumAllowedWidth - acceptButton.width - x - Theme.paddingMedium, implicitWidth)
                text: cancelText
                highlighted: true
                font {
                    pixelSize: dialog.isPortrait && !fontMetrics.smallFont ? Theme.fontSizeLarge : Theme.fontSizeMedium
                    family: Theme.fontFamilyHeading
                    weight: Font.Medium
                }
                anchors.verticalCenter: parent.verticalCenter
                truncationMode: TruncationMode.Fade
                opacity: _canGoBack && !dialogHeader._navigatingForward ? 1.0 : 0.0
                Behavior on opacity { FadeAnimation { } }
                Behavior on x { NumberAnimation { duration: 200 } }
            }
        }
        Item {
            id: extraContentPlaceholder
            x: cancelButton.preferredWidth
            width: overlay.maximumAllowedWidth - cancelButton.preferredWidth - acceptButton.width
            anchors.verticalCenter: parent.verticalCenter
            opacity: dialogHeader._navigatingBack || dialogHeader._navigatingForward ? 0.0 : 1.0
            Behavior on opacity { FadeAnimation { } }
            visible: opacity > 0
        }
        BackgroundItem {
            id: acceptButton
            onClicked: dialog.accept()

            // This tries to fit in both labels, biased toward showing the acceptText if they would overlap.
            // When moving back we reveal as much of the cancel text as we can.
            // Neither button may be larger than _maxButtonSize.
            // If the accept text is longer than dialog.width/2 then we bias toward the accept text.
            // If the cancel text is longer than dialog.width/2 then we try to fit it in without clipping the acceptText
            // If both are less than dialog.width/2 then they both get dialog.width/2
            width: Math.min(_maxButtonSize,
                       Math.max(acceptLabel.implicitWidth*(reserveExtraContent?1.0:acceptLabel.opacity) + Theme.paddingLarge + Theme.horizontalPageMargin,
                           Math.min(reserveExtraContent ? 0 : overlay.maximumAllowedWidth / 2, overlay.maximumAllowedWidth - cancelButton.preferredWidth)))

            height: overlay.height
            anchors.right: parent.right
            anchors.rightMargin: overlay.rightCutoutInset
            enabled: acceptText !== ""
            opacity: !dialogHeader.dialog || dialogHeader.dialog.canAccept ? 1.0 : Theme.opacityLow
            Behavior on opacity { FadeAnimation { } }

            down: false
            highlighted: false
            highlightedColor: "transparent"
            Binding {
                when: overlay.active && acceptButton.opacity != Theme.opacityLow
                target: pageStack._pageStackIndicator
                property: "forwardIndicatorHighlighted"
                value: (pageStack._pageStackIndicator && pageStack._pageStackIndicator.forwardIndicatorDown) || acceptButton.containsMouse
            }

            Label {
                id: acceptLabel

                objectName: "labelAcceptDialog"
                // Don't allow the label to extend over the page stack indicator
                width: acceptButton.width - Theme.paddingLarge - Theme.horizontalPageMargin
                text: acceptText
                highlighted: true
                truncationMode: TruncationMode.Fade
                font {
                    pixelSize: dialog.isPortrait && !fontMetrics.smallFont ? Theme.fontSizeLarge : Theme.fontSizeMedium
                    family: Theme.fontFamilyHeading
                    weight: Font.Medium
                }
                anchors {
                    right: parent.right
                    // |text|pad-large|indicator
                    rightMargin: dialogHeader.rightMargin
                    verticalCenter: parent.verticalCenter
                }

                // TODO: remove rich text format once QTBUG-40161 has been solved
                textFormat: Text.RichText
                horizontalAlignment: Qt.AlignRight
                opacity: dialogHeader._navigatingBack ? 0.0 : 1.0
                Behavior on opacity { FadeAnimation { } }
                Behavior on anchors.rightMargin { NumberAnimation { duration: 200 } }
            }
        }

        Rectangle {
            anchors.bottom: parent.bottom
            width: parent.width
            height: Theme._lineWidth
            color: dialogHeader.palette.primaryColor
            opacity: Theme.opacityFaint
            visible: dialogHeader.divider
        }
    }

    Label {
        id: titleText
        y: overlay.height + Theme.paddingMedium + overlay.topInset
        x: leftMargin
        width: parent.width - leftMargin - rightMargin
        font.pixelSize: Theme.fontSizeExtraLarge
        wrapMode: Text.Wrap
        color: dialogHeader.palette.highlightColor
        opacity: text.length > 0 ? 1.0 : 0.0
        Behavior on opacity { FadeAnimation { } }
    }

    // for testing
    function _headerText() {
        return acceptLabel.text
    }
    function _titleText() {
        return titleText.text
    }
}
