/****************************************************************************************
**
** SPDX-FileCopyrightText: 2014 Jolla Ltd
** SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.2
import Sailfish.Silica 1.0

SilicaItem {
    id: root

    property color progressColor: palette.highlightColor
    property color backgroundColor: palette.primaryColor
    property real maximumValue: 1.0
    property real minimumValue
    property real value
    property real progressValue
    property real borderWidth: Theme.paddingSmall
    property bool pulse
    property bool inAlternateCycle
    property bool _progressValueResetting
    readonly property real _normalizedValue: (Math.max(minimumValue, Math.min(maximumValue, value)) - minimumValue)
                                            / (maximumValue - minimumValue)

    implicitWidth: Theme.itemSizeSmall
    implicitHeight: width

    on_NormalizedValueChanged: {
        const delta = progressValue - _normalizedValue

        if (delta < -0.5 && (progressValue + delta) >= 0.0) {
            progressValue = _normalizedValue - 1.0
        } else if (delta >= 0.5) {
            progressValue = _normalizedValue + 1.0
        } else {
            progressValue = _normalizedValue
        }
    }

    onProgressValueChanged: {
        if (!_progressValueResetting && (progressValue > 1.0 || progressValue < 0.0)) {
            _progressValueResetting = true
            inAlternateCycle = !inAlternateCycle
            smoothedAnimation.enabled = false
            progressValue = progressValue % 1.0 + (progressValue < 0.0 ? 1.0 : 0.0)
            smoothedAnimation.enabled = true
            progressValue = _normalizedValue
            _progressValueResetting = false
        }
    }

    Behavior on progressValue {
        id: smoothedAnimation

        SmoothedAnimation {
            velocity: 0.2
            duration: 200
        }
    }

    ShaderEffect {
        id: shader

        property color color: palette.highlightColor
        property color secondColor: Theme.rgba(palette.primaryColor, 0.2)
        property real pulseProgress
        readonly property real visibleValue: (root.inAlternateCycle
                                              ? (root.progressValue - 1.0)
                                              : root.progressValue) * blobCount
        readonly property real pulseDelta: 0.1
        readonly property real reverseScalingFactor: 1.0 / (1.0 + pulseDelta)
        readonly property int blobCount: {
            if (root.width >= Theme.iconSizeExtraLarge) {
                return 12
            }

            if (root.width >= Theme.iconSizeMedium) {
                return 10
            }

            if (root.width >= Theme.iconSizeSmallPlus) {
                return 8
            }

            return 6
        }
        readonly property real center: {
            if (root.width >= Theme.iconSizeExtraLarge) {
                return (128.0 * 0.5 - 5.0 - 12.0 * 0.5) / 128.0
            }

            if (root.width >= Theme.iconSizeLauncher) {
                return (86.0 * 0.5 - 3.0 - 10.0 * 0.5) / 86.0
            }

            if (root.width >= Theme.iconSizeMedium) {
                return (64.0 * 0.5 - 3.0 - 8.0 * 0.5) / 64.0
            }

            if (root.width >= Theme.iconSizeSmallPlus) {
                return (48.0 * 0.5 - 3.0 - 6.0 * 0.5) / 48.0
            }

            return (32.0 * 0.5 - 3.0 - 4.0 * 0.5) / 32.0
        }
        readonly property real startSize: {
            if (root.width >= Theme.iconSizeExtraLarge) {
                return 12.0 / 128.0
            }

            if (root.width >= Theme.iconSizeLauncher) {
                return 10.0 / 86.0
            }

            if (root.width >= Theme.iconSizeMedium) {
                return 8.0 / 64.0
            }

            if (root.width >= Theme.iconSizeSmallPlus) {
                return 6.0 / 48.0
            }

            return 4.0 / 32.0
        }
        readonly property real endSize: {
            if (root.width >= Theme.iconSizeExtraLarge) {
                return 18.0 / 128.0
            }

            if (root.width >= Theme.iconSizeLauncher) {
                return 14.0 / 86.0
            }

            if (root.width >= Theme.iconSizeMedium) {
                return 12.0 / 64.0
            }

            if (root.width >= Theme.iconSizeSmallPlus) {
                return 10.0 / 48.0
            }

            return 8.0 / 32.0
        }

        anchors.centerIn: parent

        width: root.width * (1.0 + pulseDelta)
        height: width

        vertexShader: "
            #define COUNT      12 // Should be equal to fragmentShader.COUNT and busyIndicator.count
            #define PI2        1.57079632679 // pi / 2
            #define TAU        6.28318530718 // pi * 2

            precision lowp float;

            uniform highp mat4 qt_Matrix;
            attribute highp vec4 qt_Vertex;
            attribute highp vec2 qt_MultiTexCoord0;

            varying highp vec2 coord;
            varying vec2 pointArray[COUNT];
            varying float valueArray[COUNT];
            varying float sizeArray[COUNT];
            uniform lowp int blobCount;
            uniform float center;
            uniform float pulseProgress;
            uniform float visibleValue;
            uniform float pulseDelta;
            uniform float reverseScalingFactor;
            uniform float startSize;
            uniform float endSize;

            void main() {
                coord = qt_MultiTexCoord0 - 0.5;
                float count = float(blobCount);

                // Looping
                float pulseScale = pulseProgress > 1.0 ? 2.0 - pulseProgress : pulseProgress;
                // To easeInOutQuad
                pulseScale = pulseScale < 0.5 ? 2.0 * pulseScale * pulseScale : 1.0 - pow(-2.0 * pulseScale + 2.0, 2.0) * 0.5;
                // To pulseDelta
                pulseScale = pulseScale * pulseDelta + 1.0;

                for (int i = 0; i < blobCount; i++) {
                    float fi = float(i);
                    float angle = (fi / count * TAU) - PI2 + (0.5 / count * TAU);
                    pointArray[i] = vec2(cos(angle), sin(angle)) * center * pulseScale * reverseScalingFactor;
                    valueArray[i] = clamp(visibleValue - fi, 0.0, 1.0) + clamp(fi - count + 1.0 - visibleValue, 0.0, 1.0);
                    sizeArray[i] = (startSize + (endSize - startSize) * valueArray[i]) * 0.5 * pulseScale * reverseScalingFactor;
                }

                gl_Position = qt_Matrix * qt_Vertex;
            }
        "

        fragmentShader: "
            #define COUNT 12 // Should be equal to vertexShader.COUNT and busyIndicator.count

            precision lowp float;

            uniform vec4 color;
            uniform vec4 secondColor;
            uniform highp float width;
            uniform float qt_Opacity;
            uniform lowp int blobCount;

            varying highp vec2 coord;
            varying vec2 pointArray[COUNT];
            varying float valueArray[COUNT];
            varying float sizeArray[COUNT];

            void main() {
                float value = 0.0;
                float colorValue = 0.0;
                for (int i = 0; i < blobCount; i++) {
                    float blob = 1.0 - clamp((length(coord - pointArray[i]) - sizeArray[i]) * width, 0.0, 1.0);
                    value += blob;
                    colorValue += float(bool(blob)) * valueArray[i];
                }
                gl_FragColor = vec4(mix(secondColor, color, colorValue * qt_Opacity)) * value * qt_Opacity;
            }
        "

        NumberAnimation {
            target: shader
            property: "pulseProgress"
            running: root.pulse && root.visible && root.opacity > 0.0 && Qt.application.active
            loops: Animation.Infinite
            alwaysRunToEnd: true
            easing.type: Easing.Linear
            from: 0.0
            to: 2.0
            duration: 1000
        }
    }
}
