/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import QtFeedback 5.0
import Sailfish.Silica 1.0
import ru.auroraos.WebView.Global 1.0
import ru.auroraos.WebView.Private 1.0

OverscrollExtension {
    id: overscrollExtension

    readonly property double gestureThreshold: Theme.dp(148)

    onDeltaYChanged: {
        if (overscrollExtension.mode === OverscrollMode.OVERSCROLL_MODE_SOUTH) {
            if (deltaY <= gestureThreshold) {
                indicator.y = -Theme.itemSizeMedium + deltaY
                indicator.time = deltaY / gestureThreshold

                if (haptics.finished)
                    haptics.finished = false
            } else if (!haptics.finished) {
                haptics.play(ThemeEffect.PressStrong)
                haptics.finished = true
            }
        }
    }

    onCompleted: {
        if (overscrollExtension.mode === OverscrollMode.OVERSCROLL_MODE_SOUTH) {
            if (deltaY >= gestureThreshold) {
                indicator.running = true
                parent.reload()
            } else {
                indicator.y = -Theme.itemSizeMedium
                indicator.time = 0.0
                indicator.visible = false

                parent.opacity = 1.0
            }
        }
    }

    onModeChanged: {
        if (newMode === OverscrollMode.OVERSCROLL_MODE_SOUTH) {
            indicator.visible = true
            haptics.finished = false

            parent.opacity = 0.6
        } else if (indicator.visible
                   && newMode === OverscrollMode.OVERSCROLL_MODE_NONE) {
            indicator.y = -Theme.itemSizeMedium
            indicator.time = 0.0
            indicator.visible = false

            parent.opacity = 1.0
        }
    }

    anchors.horizontalCenter: parent.horizontalCenter

    RefreshIndicator {
        id: indicator

        property alias animation: hideAnimation

        y: -Theme.itemSizeMedium
        time: 0.0

        SequentialAnimation {
            id: hideAnimation

            NumberAnimation {
                target: indicator
                property: "y"
                to: -Theme.itemSizeMedium
                duration: 200
            }
            ScriptAction {
                script: {
                    if (indicator.visible)
                        indicator.visible = false
                    indicator.running = false

                    parent.opacity = 1.0
                }
            }
        }
    }

    ThemeEffect {
        id: haptics

        property bool finished: false
    }

    Connections {
        // Remove when onLoadingChanged is fixed
        onLoadingProgressChanged: {
            if (parent.loadingProgress && parent.loadingProgress === 1.0
                    && indicator.running) {
                hideAnimation.start()
            }
        }

        onLoadingChanged: {
            if (!parent.loading && indicator.running) {
                hideAnimation.start()
            }
        }

        target: parent
    }
}
