// SPDX-FileCopyrightText: 2023 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#ifndef VIDEODECODERH264STREAMCAMERA_H
#define VIDEODECODERH264STREAMCAMERA_H

#include "job_queue.h"

#include <streamcamera/streamcamera-codec.h>
#include <streamcamera/streamcamera.h>

#include "api/video_codecs/video_decoder.h"
#include "rtc_base/checks.h"
#include "rtc_base/logging.h"

#include <atomic>
#include <list>
#include <memory>
#include <mutex>
#include <thread>
#include <vector>

class VideoDecoderH264StreamCamera : public webrtc::VideoDecoder
{
public:
    class VideoDecoderListenerImpl : public Aurora::StreamCamera::VideoDecoderListener
    {
    public:
        VideoDecoderListenerImpl(VideoDecoderH264StreamCamera &decoder)
            : decoder_(decoder)
        {}

        ~VideoDecoderListenerImpl() override = default;

        void onDecodedYCbCrFrame(const Aurora::StreamCamera::YCbCrFrame *frame) override
        {
            decoder_.onDecodedYCbCrFrame(frame);
        }

        void onDecodedGraphicBuffer(
            std::shared_ptr<Aurora::StreamCamera::GraphicBuffer> buffer) override
        {
            decoder_.onDecodedGraphicBuffer(buffer);
        }

        void onDecoderParameterChanged(Aurora::StreamCamera::VideoDecoderParameter param,
                                       const std::string &value) override
        {
            decoder_.onDecoderParameterChanged(param, value);
        }

        void onDecoderError(const std::string &errorDescription) override
        {
            RTC_LOG_F(LS_ERROR) << errorDescription;
        }

        void onDecoderEOS() override { RTC_LOG_F(LS_INFO) << ""; }

        VideoDecoderH264StreamCamera &decoder_;
    };

    struct EncodedMetaData
    {
        int width;
        int height;
        uint64_t timestampUs;
        Aurora::StreamCamera::FrameType frameType;
        bool inDecoder = false;
    };

    struct DataList
    {
        struct DataItem
        {
            DataList *owner_;
            std::list<DataItem>::iterator it_;
            EncodedMetaData metaData_;
            std::vector<uint8_t> data_;
        };

        template<typename... Args>
        void emplace_back(EncodedMetaData metaData, Args &&... args)
        {
            list_.emplace_back(DataItem{this, {}, metaData, std::vector<uint8_t>{args...}});
            list_.back().it_ = std::prev(list_.end());
        }

        auto erase(std::list<DataItem>::iterator it)
        {
            RTC_DCHECK(!list_.empty());
            return list_.erase(it);
        }

        auto end() { return list_.end(); }

        auto notInDecoder()
        {
            return std::find_if(list_.begin(), list_.end(), [](const auto &item) {
                return !item.metaData_.inDecoder;
            });
        }

        size_t size() { return list_.empty() ? 0 : list_.size(); }
        bool empty() { return list_.empty(); }

        std::list<DataItem> list_;
        std::mutex mutex_;
    };

    static std::unique_ptr<VideoDecoderH264StreamCamera> Create()
    {
        return std::make_unique<VideoDecoderH264StreamCamera>();
    }

    static void decoder_data_release_cb(void *data_item_ptr)
    {
        auto item = reinterpret_cast<DataList::DataItem *>(data_item_ptr);
        RTC_DCHECK(item);
        RTC_DCHECK(item->owner_);
        DataList &list = *item->owner_;
        std::scoped_lock lock(list.mutex_);
        list.erase(item->it_);
    }

    VideoDecoderH264StreamCamera();
    ~VideoDecoderH264StreamCamera() override;

    // Prepares decoder to handle incoming encoded frames. Can be called multiple
    // times, in such case only latest `settings` are in effect.
    bool Configure(const Settings &settings) override;

    int32_t Decode(const webrtc::EncodedImage &input_image,
                   bool missing_frames,
                   int64_t render_time_ms) override;
    int32_t RegisterDecodeCompleteCallback(webrtc::DecodedImageCallback *callback) override;
    int32_t Release() override;
    webrtc::VideoDecoder::DecoderInfo GetDecoderInfo() const override;
    // Deprecated, use GetDecoderInfo().implementation_name instead.
    const char *ImplementationName() const override;

private:
    bool ConfigureDo();
    void DecodeDo();
    // Streamcamera doc say:
    // Some devices cannot produce graphic buffers, so they call
    // onDecodedYCbCrFrame(). The buffer is not shareable and the data will
    // become invalid after the application returns from this callback.
    void onDecodedYCbCrFrame(const Aurora::StreamCamera::YCbCrFrame *camera_frame);
    void onDecodedGraphicBuffer(std::shared_ptr<Aurora::StreamCamera::GraphicBuffer> buffer);
    void onDecoderParameterChanged(Aurora::StreamCamera::VideoDecoderParameter param,
                                   const std::string &value);

private:
    Aurora::StreamCamera::CodecManager &codec_manager_;
    std::shared_ptr<Aurora::StreamCamera::VideoDecoder> decoder_;
    std::vector<Aurora::StreamCamera::PixelFormat> pixel_format_list_;
    VideoDecoderListenerImpl listener_;
    webrtc::DecodedImageCallback *decoded_callback_ = nullptr;
    std::mutex decoded_callback_mutex_;
    Settings settings_;
    int width_ = 0;
    int height_ = 0;
    DataList dataList_;
    job_queue_t job_queue_;
    std::thread thread_;
    std::atomic<bool> run_ = ATOMIC_VAR_INIT(false);

    // Number of hardware decoders is limited.
    // Check how many decoders WebRTC creates
    static std::atomic<size_t> instance_count_;
};

#endif // VIDEODECODERH264STREAMCAMERA_H
