/**
 * SPDX-FileCopyrightText: 2007-2014 Intel Corporation
 * SPDX-FileCopyrightText: 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: GPL-2.0-only
 */

#ifndef __CONNMAN_SERVICE_H
#define __CONNMAN_SERVICE_H

#include <stdbool.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * SECTION:service
 * @title: Service premitives
 * @short_description: Functions for handling services
 */

enum connman_service_type {
	CONNMAN_SERVICE_TYPE_UNKNOWN   = 0,
	CONNMAN_SERVICE_TYPE_SYSTEM    = 1,
	CONNMAN_SERVICE_TYPE_ETHERNET  = 2,
	CONNMAN_SERVICE_TYPE_WIFI      = 3,
	CONNMAN_SERVICE_TYPE_BLUETOOTH = 4,
	CONNMAN_SERVICE_TYPE_CELLULAR  = 5,
	CONNMAN_SERVICE_TYPE_GPS       = 6,
	CONNMAN_SERVICE_TYPE_VPN       = 7,
	CONNMAN_SERVICE_TYPE_GADGET    = 8,
	CONNMAN_SERVICE_TYPE_P2P       = 9,
};
#define	MAX_CONNMAN_SERVICE_TYPES        10
typedef enum connman_service_type CServiceType;

enum connman_service_security {
	CONNMAN_SERVICE_SECURITY_UNKNOWN = 0,
	CONNMAN_SERVICE_SECURITY_NONE    = 1,
	CONNMAN_SERVICE_SECURITY_WEP     = 2,
	CONNMAN_SERVICE_SECURITY_PSK     = 3,
	CONNMAN_SERVICE_SECURITY_8021X   = 4,
	CONNMAN_SERVICE_SECURITY_WPA     = 8,
	CONNMAN_SERVICE_SECURITY_RSN     = 9,
};
typedef enum connman_service_security CServiceSecurityType;

enum connman_service_state {
	CONNMAN_SERVICE_STATE_UNKNOWN       = 0,
	CONNMAN_SERVICE_STATE_IDLE          = 1,
	CONNMAN_SERVICE_STATE_ASSOCIATION   = 2,
	CONNMAN_SERVICE_STATE_CONFIGURATION = 3,
	CONNMAN_SERVICE_STATE_READY         = 4,
	CONNMAN_SERVICE_STATE_ONLINE        = 5,
	CONNMAN_SERVICE_STATE_DISCONNECT    = 6,
	CONNMAN_SERVICE_STATE_FAILURE       = 7,
};
typedef enum connman_service_state CServiceState;

enum connman_service_error {
	CONNMAN_SERVICE_ERROR_UNKNOWN        = 0,
	CONNMAN_SERVICE_ERROR_OUT_OF_RANGE   = 1,
	CONNMAN_SERVICE_ERROR_PIN_MISSING    = 2,
	CONNMAN_SERVICE_ERROR_DHCP_FAILED    = 3,
	CONNMAN_SERVICE_ERROR_CONNECT_FAILED = 4,
	CONNMAN_SERVICE_ERROR_LOGIN_FAILED  = 5,
	CONNMAN_SERVICE_ERROR_AUTH_FAILED    = 6,
	CONNMAN_SERVICE_ERROR_INVALID_KEY    = 7,
};
typedef enum connman_service_error CServiceError;

enum connman_service_proxy_method {
	CONNMAN_SERVICE_PROXY_METHOD_UNKNOWN     = 0,
	CONNMAN_SERVICE_PROXY_METHOD_DIRECT      = 1,
	CONNMAN_SERVICE_PROXY_METHOD_MANUAL      = 2,
	CONNMAN_SERVICE_PROXY_METHOD_AUTO        = 3,
};
typedef enum connman_service_proxy_method CServiceProxyMethod;

enum connman_service_connect_reason {
	CONNMAN_SERVICE_CONNECT_REASON_NONE	= 0,
	CONNMAN_SERVICE_CONNECT_REASON_AUTO	= 1,
	CONNMAN_SERVICE_CONNECT_REASON_USER	= 2,
	CONNMAN_SERVICE_CONNECT_REASON_SESSION	= 3,
	CONNMAN_SERVICE_CONNECT_REASON_DEFAULT_CHANGE = 4,
};
typedef enum connman_service_connect_reason CServiceConnectReason;

struct connman_service;
struct connman_network;

struct connman_service *connman_service_create(void);

#define connman_service_ref(service) \
	connman_service_ref_debug(service, __FILE__, __LINE__, __func__)

#define connman_service_unref(service) \
	connman_service_unref_debug(service, __FILE__, __LINE__, __func__)

struct connman_service *
connman_service_ref_debug(struct connman_service *service,
			const char *file, int line, const char *caller);
void connman_service_unref_debug(struct connman_service *service,
			const char *file, int line, const char *caller);

enum connman_service_type connman_service_get_type(struct connman_service *service);
enum connman_service_state connman_service_get_state(struct connman_service *service);
char *connman_service_get_interface(struct connman_service *service);

const char *connman_service_get_identifier(struct connman_service *service);
const char *connman_service_get_domainname(struct connman_service *service);
char **connman_service_get_nameservers(struct connman_service *service);
char **connman_service_get_timeservers_config(struct connman_service *service);
char **connman_service_get_timeservers(struct connman_service *service);
void connman_service_set_proxy_method(struct connman_service *service, enum connman_service_proxy_method method);
enum connman_service_proxy_method connman_service_get_proxy_method(struct connman_service *service);
char **connman_service_get_proxy_servers(struct connman_service *service);
char **connman_service_get_proxy_excludes(struct connman_service *service);
const char *connman_service_get_proxy_url(struct connman_service *service);
const char *connman_service_get_proxy_autoconfig(struct connman_service *service);
bool connman_service_get_favorite(struct connman_service *service);
bool connman_service_get_saved(struct connman_service *service);
bool connman_service_get_autoconnect(struct connman_service *service);
bool connman_service_set_autoconnect(struct connman_service *service,
							bool autoconnect);
int connman_service_get_available_count(enum connman_service_type type);
int connman_service_get_online_check_passed_amount(struct connman_service *service,
							enum connman_ipconfig_type type);

/* Return non-zero value to terminate the loop, zero to continue */
typedef int (* connman_service_iterate_cb) (struct connman_service *service,
							void *user_data);
int connman_service_iterate_services(connman_service_iterate_cb cb,
							void *user_data);

struct connman_service *connman_service_get_default(void);
struct connman_service *connman_service_lookup_from_network(struct connman_network *network);
void connman_service_update_strength_from_network(struct connman_network *network);
struct connman_service *connman_service_lookup_from_identifier(const char* identifier);

void connman_service_create_ip4config(struct connman_service *service,
								int index);
void connman_service_create_ip6config(struct connman_service *service,
								int index);
#ifdef __cplusplus
}
#endif

#endif /* __CONNMAN_SERVICE_H */
