/*
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <memory>
#include <QObject>

#include "browserextension.h"
#include "enums/types.h"

namespace Aurora {
namespace WebView {

/**
 * @class IOverscrollExtension
 * @brief extension that provides overscroll information
 * 
 * This extension allows the developer to receive overscroll events 
 * in order to implement custom behaviour, such as processing a pull 
 * to refresh gesture.
 */
class IOverscrollExtension : public QObject, public Cef::BrowserExtension
{
    Q_OBJECT
public:
    IOverscrollExtension(Cef::Browser &browser)
        : Cef::BrowserExtension(browser)
    {}
    virtual ~IOverscrollExtension() = default;

    /// horizontal overscroll value
    virtual float deltaX() const = 0;
    /// vertical overscroll value
    virtual float deltaY() const = 0;
    /// overscroll mode
    ///
    /// For example, if the user scrolls the page down to the end and makes a gesture
    /// to scroll it further down, the mode will be
    /// Aurora::WebView::OverscrollMode::Type::OVERSCROLL_MODE_NORTH.
    virtual Aurora::WebView::OverscrollMode::Type mode() const = 0;
    /// previous overscroll mode in case the mode was changed during the overscroll process
    virtual Aurora::WebView::OverscrollMode::Type oldMode() const = 0;

signals:
    /**
     * @brief emitted when horizontal overscroll value is changed
     * 
     * @param deltaX horizontal overscroll value
     */
    void deltaXChanged(float deltaX);
    /**
     * @brief emitted when vertical overscroll value is changed
     * 
     * @param deltaY vertical overscroll value
     */
    void deltaYChanged(float deltaY);
    /**
     * @brief emitted when overscroll is completed
     * 
     * This signal is emitted when overscroll threshold is reached and gesture 
     * is finished (finger is lifted up).
     * 
     * @param mode overscroll mode
     */
    void completed(Aurora::WebView::OverscrollMode::Type mode);
    /**
     * @brief emitted when overscroll mode is changed during overscroll gesture 
     * 
     * @param oldMode the mode from which the change was made
     * @param newMode new overscroll mode
     */
    void modeChanged(
        Aurora::WebView::OverscrollMode::Type oldMode,
        Aurora::WebView::OverscrollMode::Type newMode);
};

} // namespace WebView
} // namespace Aurora
