/*
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <QObject>
#include <QString>

namespace Aurora {
namespace WebView {

/**
 * @class ContextMenuParams
 * @brief contains context menu fields.
 */
class ContextMenuParams
{
    Q_GADGET

    /// Aurora::WebView::ContextMenuTypeFlags
    Q_PROPERTY(int typeFlags READ typeFlags)
    /// Aurora::WebView::ContextMenuMediaType
    Q_PROPERTY(int mediaType READ mediaType)
    /// Aurora::WebView::ContextMenuMediaStateFlags
    Q_PROPERTY(int mediaStateFlags READ mediaStateFlags)
    /// Aurora::WebView::ContextMenuEditStateFlags
    Q_PROPERTY(int editStateFlags READ editStateFlags)

    /// A list of default supported available actions
    /// from Aurora::WebView::MenuId::MENU_ID_USER_COPY_LINK
    /// to Aurora::WebView::MenuId::MENU_ID_USER_COPY_IMAGE
    Q_PROPERTY(QVariantList availableActions READ availableActions)

    /// url of a link
    Q_PROPERTY(QString linkUrl READ linkUrl)
    /// link url to be used in "Copy Link Address"
    Q_PROPERTY(QString unfilteredLinkUrl READ unfilteredLinkUrl)
    /// url of a source media, if any. For example, of an image
    Q_PROPERTY(QString sourceUrl READ sourceUrl)
    /// title text
    Q_PROPERTY(QString titleText READ titleText)
    /// selection text
    Q_PROPERTY(QString selectionText READ selectionText)
public:
    ContextMenuParams(
        int menuTypeFlags,
        int menuMediaType,
        int menuMediaStateFlags,
        int menuEditStateFlags,
        QVariantList availableActions,
        QString linkUrl,
        QString unfilteredLinkUrl,
        QString sourceUrl,
        QString titleText,
        QString selectionText)
        : m_menuTypeFlags(menuTypeFlags)
        , m_menuMediaType(menuMediaType)
        , m_menuMediaStateFlags(menuMediaStateFlags)
        , m_menuEditStateFlags(menuEditStateFlags)
        , m_availableActions(availableActions)
        , m_linkUrl(linkUrl)
        , m_unfilteredLinkUrl(unfilteredLinkUrl)
        , m_sourceUrl(sourceUrl)
        , m_titleText(titleText)
        , m_selectionText(selectionText)
    {}

    ContextMenuParams()
        : m_menuTypeFlags(-1)
        , m_menuMediaType(-1)
        , m_menuMediaStateFlags(-1)
        , m_menuEditStateFlags(-1)
        , m_availableActions(QVariantList())
        , m_linkUrl("")
        , m_unfilteredLinkUrl("")
        , m_sourceUrl("")
        , m_titleText("")
        , m_selectionText("")
    {}
    ContextMenuParams(const ContextMenuParams &other) = default;
    virtual ~ContextMenuParams() = default;

    int typeFlags() const { return m_menuTypeFlags; }
    int mediaType() const { return m_menuMediaType; }
    int mediaStateFlags() const { return m_menuMediaStateFlags; }
    int editStateFlags() const { return m_menuEditStateFlags; }

    QVariantList availableActions() const { return m_availableActions; }

    QString linkUrl() const { return m_linkUrl; }
    QString unfilteredLinkUrl() const { return m_unfilteredLinkUrl; }
    QString sourceUrl() const { return m_sourceUrl; }
    QString titleText() const { return m_titleText; }
    QString selectionText() const { return m_selectionText; }

private:
    int m_menuTypeFlags;       // Aurora::WebView::ContextMenuTypeFlags
    int m_menuMediaType;       // Aurora::WebView::ContextMenuMediaType
    int m_menuMediaStateFlags; // Aurora::WebView::ContextMenuMediaStateFlags
    int m_menuEditStateFlags;  // Aurora::WebView::ContextMenuEditStateFlags

    QVariantList m_availableActions; // Aurora::WebView::MenuId

    QString m_linkUrl;
    QString m_unfilteredLinkUrl;
    QString m_sourceUrl;
    QString m_titleText;
    QString m_selectionText;
};

} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::ContextMenuParams)
