/*
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <QDateTime>
#include <QObject>
#include <QString>

#include "enums/types.h"

class CefCookieTraits;
template<class T>
class CefStructBase;

typedef CefStructBase<CefCookieTraits> CefCookie;

namespace Aurora {
namespace WebView {

/**
 * @class Cookie
 * @brief Represents a cookie file and stores its properties.
 */
class Cookie
{
    Q_GADGET

    /// Cookie name
    Q_PROPERTY(QString name READ name)
    /// Value that is stored in the cookie
    Q_PROPERTY(QString value READ value)
    /// Domain name
    Q_PROPERTY(QString domain READ domain)
    /// Path
    Q_PROPERTY(QString path READ path)

    /// true when the cookie is only sent for HTTPS requests
    Q_PROPERTY(bool secure READ secure)
    /// true when the cookie is only sent for HTTP requests
    Q_PROPERTY(bool httpOnly READ httpOnly)
    /// if true, expires is valid and can be used
    Q_PROPERTY(bool hasExpirationDate READ hasExpirationDate)

    /// Cookie creation date
    Q_PROPERTY(QDateTime creationDate READ creationDate)
    /// Date of the last cookie access
    Q_PROPERTY(QDateTime lastAccess READ lastAccess)
    ///
    /// Expiration date.
    ///
    /// Valid only if hasExpirationDate is true
    ///
    Q_PROPERTY(QDateTime expires READ expires)

    /// Cookie same site property (Aurora::WebView::Cookies::CookieSameSite)
    Q_PROPERTY(int sameSite READ sameSite)
    /// Cookie priority property (Aurora::WebView::Cookies::CookiePriority)
    Q_PROPERTY(int priority READ priority)
public:
    /**
     * @brief Constructs an empty Cookie object
     */
    Cookie();
    /**
     * @brief Default destructor
     */
    virtual ~Cookie() = default;
    /**
     * @brief Default copy constructor
     */
    Cookie(const Cookie &other) = default;

    /**
     * @brief Converts CefCookie into Cookie
     */
    Cookie(const CefCookie &other);

    Cookie(
        QString name,
        QString value,
        QString domain,
        QString path,
        bool secure,
        bool httpOnly,
        bool hasExpirationDate,
        QDateTime creationDate,
        QDateTime lastAccess,
        QDateTime expires,
        Cookies::CookieSameSite sameSite,
        Cookies::CookiePriority priority);

    /// Cookie name
    QString name() const;
    /// Value that is stored in the cookie
    QString value() const;
    /// Domain name
    QString domain() const;
    /// Path
    QString path() const;

    /// true when the cookie is only sent for HTTPS requests
    bool secure() const;
    /// true when the cookie is only sent for HTTP requests
    bool httpOnly() const;
    /// if true, expires is valid and can be used
    bool hasExpirationDate() const;

    /// Cookie creation date
    QDateTime creationDate() const;
    /// Date of the last cookie access
    QDateTime lastAccess() const;
    ///
    /// Expiration date.
    ///
    /// Valid only if hasExpirationDate is true
    ///
    QDateTime expires() const;

    /// Cookie same site property (Aurora::WebView::Cookies::CookieSameSite)
    int sameSite() const;
    /// Cookie priority property (Aurora::WebView::Cookies::CookiePriority)
    int priority() const;

private:
    const QString m_name;
    const QString m_value;
    const QString m_domain;
    const QString m_path;

    const bool m_secure;
    const bool m_httpOnly;
    const bool m_hasExpirationDate;

    const QDateTime m_creation;
    const QDateTime m_lastAccess;
    const QDateTime m_expires;

    const int m_sameSite;
    const int m_priority;
};
} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::Cookie)
