/**
 * SPDX-FileCopyrightText: Copyright 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <QObject>
#include <QDBusAbstractAdaptor>
#include <QDBusObjectPath>

#include <callserviceglobal.h>

namespace ru {
namespace auroraos {
namespace call {

class CallManager;
class CallPrivate;

class CALLSERVICE_EXPORT Call : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QString callId READ callId CONSTANT);
    Q_PROPERTY(QString uri READ uri CONSTANT);
    Q_PROPERTY(bool incoming READ incoming CONSTANT);
    Q_PROPERTY(bool holdable READ holdable CONSTANT);
    Q_PROPERTY(QString remoteHandle READ remoteHandle CONSTANT);
    Q_PROPERTY(QString remoteName READ remoteName CONSTANT);
    Q_PROPERTY(QString localHandle READ localHandle CONSTANT);
    Q_PROPERTY(QString localName READ localName CONSTANT);
    Q_PROPERTY(qint64 initiatedAt READ initiatedAt CONSTANT);
    Q_PROPERTY(qint64 acceptedAt READ acceptedAt);
    Q_PROPERTY(Status status READ status NOTIFY statusChanged);
    Q_PROPERTY(bool active READ active NOTIFY activeChanged);

public:
    enum Status {
        Unknown,
        Disconnected,
        Dialing,
        Ringing,
        Rejecting,
        Accepting,
        Active,
        Held,
    };

    Q_ENUM(Status)

public:
    explicit Call(CallManager *manager, const QString &id, const QVariantMap &parameters, QObject *parent = nullptr);
    virtual ~Call();

    QDBusObjectPath objectPath() const;

    void addInterfaces(const QList<QObject*> &handlers);

    QString callId() const;

    QString uri() const;
    bool incoming() const;
    bool holdable() const;
    QString remoteHandle() const;
    QString remoteName() const;
    QString localHandle() const;
    QString localName() const;
    qint64 initiatedAt() const;
    qint64 acceptedAt() const;

    Status status() const;
    bool active() const;

public slots:
    void setStatus(Status status);

signals:
    void accepted();
    void rejected();
    void disconnected();
    void heldChanged(bool held);
    void statusChanged(Status status);
    void activeChanged(bool active);

private:
    Q_DISABLE_COPY(Call)
    Q_DECLARE_PRIVATE(Call)
    CallPrivate *d_ptr;

    friend class CallManager;
    friend class CallManagerDBus;
};

} // namespace call
} // namespace auroraos
} // namespace ru
