/*
 * SPDX-FileCopyrightText: Copyright 2024-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#pragma once

#include <memory>
#include <QObject>

namespace Aurora {
namespace WebView {

class HttpAuthRequestInternal;

/**
 * @class HttpAuthRequest
 * @brief Represents an HTTP auth request
 * 
 * Provides details of the request, e.g., url of request origin, and allows authenticating
 * or cancelling the request.
 */
class HttpAuthRequest
{
    Q_GADGET

    /// url of request origin
    Q_PROPERTY(QString originUrl READ originUrl)
    /// hostname
    Q_PROPERTY(QString host READ host)
    /// is the realm of the challenge and might be empty
    Q_PROPERTY(QString realm READ realm)
    /// authentication scheme (e.g., "basic" or "digest")
    ///
    /// empty if the source of the request is an FTP server
    Q_PROPERTY(QString scheme READ scheme)
    /// indicates whether the host is a proxy server
    Q_PROPERTY(bool isProxy READ isProxy)
    /// port number
    Q_PROPERTY(int port READ port)
public:
    HttpAuthRequest();
    HttpAuthRequest(const HttpAuthRequest &other) = default;
    virtual ~HttpAuthRequest() = default;

    /// @brief call to attempt to authenticate
    /// @param username username
    /// @param password password of the user
    Q_INVOKABLE virtual void authenticate(const QString username, const QString password);
    /// @brief call to cancel authentication
    Q_INVOKABLE virtual void cancel();

    virtual QString originUrl() const;
    virtual QString host() const;
    virtual QString realm() const;
    virtual QString scheme() const;
    virtual bool isProxy() const;
    virtual int port() const;
    virtual bool isValid() const;

private:
    HttpAuthRequest(std::shared_ptr<HttpAuthRequestInternal> httpAuthRequestInternal);

    std::shared_ptr<HttpAuthRequestInternal> m_d;

    friend class HttpAuthRequestInternal;
};

} // namespace WebView
} // namespace Aurora

Q_DECLARE_METATYPE(Aurora::WebView::HttpAuthRequest)
