/****************************************************************************************
**
** SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
****************************************************************************************/

import QtQuick 2.5
import Sailfish.Silica 1.0

Item {
    id: root

    property bool checked
    property bool busy
    property bool highlighted
    property bool textSwitch

    property bool _diskPulsating
    property real _diskCheckOpacity
    property color _diskColor
    readonly property real _diskPulseOpacity: Math.min(1.0, Math.max(0.0, (shaderEffect.diskRadius - 0.5) * 4.0 * 0.7 + 0.3))
    readonly property bool _busy: root.busy && Qt.application.active
    property alias _shaderOpacity: shaderEffect.diskOpacity

    width: Theme.itemSizeExtraSmall
    height: Theme.itemSizeExtraSmall

    Binding {
        when: root.checked
        target: root
        property: "_diskColor"
        value: {
            if (root.enabled) {
                // "enabled_checked"
                if (!root.highlighted) {
                    return palette.highlightColor
                }
                // "enabled_checked_hover"
                return Qt.tint(palette.highlightColor, Theme.rgba(palette.highlightBackgroundColor, 0.3))
            }
            // "disabled_checked"
            return Theme.rgba(palette.primaryColor, Theme.opacityLow)
        }
    }

    Behavior on _diskCheckOpacity {
        NumberAnimation {
            id: _diskCheckOpacityAnimation

            duration: 200
            easing.type: Easing.InOutQuad
        }
    }

    ShaderEffect {
        id: shaderEffect

        readonly property color borderColor: {
            if (diskRadius == 1.0 && diskOpacity == 1.0) {
                return diskColor
            }

            if (root.enabled) {
                // "enabled"
                if (!root.highlighted) {
                    return palette.primaryColor
                }
                // "enabled_hover"
                return palette.highlightColor
            }
            // "disabled"
            if (!root.checked) {
                return Theme.rgba(palette.primaryColor, Theme.opacityLow)
            }
            // "disabled_checked"
            return "transparent"
        }
        property real diskRadius
        readonly property real borderSize: Theme.dp(2)
        readonly property color diskColor: Theme.rgba(root._diskColor, 1.0)
        readonly property real diskOpacity: root._diskColor.a * root._diskCheckOpacity * root._diskPulseOpacity
        readonly property real texSize: width
        readonly property variant mark: markItem

        anchors.verticalCenter: parent.verticalCenter
        x: root.textSwitch ? 0 : (parent.width - width) * 0.5

        width: Theme.iconSizeSmall
        height: width

        fragmentShader:"
            uniform lowp float texSize;
            varying highp vec2 qt_TexCoord0;
            uniform lowp vec4 diskColor;
            uniform lowp float diskOpacity;
            uniform lowp float qt_Opacity;
            uniform mediump float diskRadius;
            uniform sampler2D mark;
            uniform lowp vec4 borderColor;
            uniform lowp float borderSize;

            void main() {
                highp float center = texSize * 0.5;
                highp float dist = length(qt_TexCoord0 * texSize - center);
                highp float alpha = borderColor.a * clamp(dist - center + borderSize + 1.0, 0.0, 1.0);
                highp vec3 borderTexColor = borderColor.rgb * alpha;
                highp vec4 diskTexColor = diskColor * (1.0 - texture2D(mark, qt_TexCoord0).a);
                diskTexColor.a *= diskOpacity * (1.0 - clamp(dist - center * diskRadius + 1.0, 0.0, 1.0));
                alpha = diskTexColor.a + alpha * (1.0 - diskTexColor.a);
                highp vec3 texColor = (diskTexColor.rgb * diskTexColor.a + borderTexColor * (1.0 - diskTexColor.a)) / alpha;
                gl_FragColor = vec4(texColor, 1.0) * alpha * (1.0 - clamp(dist - center + 1.0, 0.0, 1.0)) * qt_Opacity;
            }"

        Behavior on diskRadius {
            id: diskRadiusBehavior

            NumberAnimation { duration: 500; easing.type: Easing.InOutQuad }
        }

        SequentialAnimation {
            id: pulseAnimation

            property bool rise

            loops: Animation.Infinite
            PropertyAnimation {
                target: shaderEffect
                property: "diskRadius"
                to: pulseAnimation.rise ? 0.75 : 0.5
                duration: 500
                easing.type: Easing.InOutSine
            }
            PropertyAnimation {
                target: shaderEffect
                property: "diskRadius"
                to: pulseAnimation.rise ? 0.5 : 0.75
                duration: 500
                easing.type: Easing.InOutSine
            }
        }

        states: [
            State {
                name: "checked"
                when: !root._busy && root.checked
                StateChangeScript {
                    script: {
                        pulseAnimation.stop()

                        if (root._diskCheckOpacity == 0.0) {
                            diskRadiusBehavior.enabled = false
                            shaderEffect.diskRadius = 1.0
                        }

                        _diskCheckOpacityAnimation.duration = root._diskPulsating ? 500 : 200
                        root._diskCheckOpacity = 1.0
                        diskRadiusBehavior.enabled = true
                        shaderEffect.diskRadius = 1.0
                        root._diskPulsating = false
                    }
                }

            }, State {
                name: "unchecked"
                when: !root._busy && !root.checked
                StateChangeScript {
                    script: {
                        pulseAnimation.stop()
                        _diskCheckOpacityAnimation.duration = root._diskPulsating ? 500 : 200
                        root._diskCheckOpacity = 0.0
                        diskRadiusBehavior.enabled = true
                        shaderEffect.diskRadius = root._diskPulsating ? 0.5 : 1.0
                        root._diskPulsating = false
                    }
                }

            }, State {
                name: "pulse"
                when: root._busy && Qt.application.active
                StateChangeScript {
                    script: {
                        diskRadiusBehavior.enabled = false

                        if (root._diskCheckOpacity == 0.0) {
                            shaderEffect.diskRadius = root.checked ? 1.0 : 0.5
                        }

                        root._diskCheckOpacity = 1.0
                        pulseAnimation.rise = shaderEffect.diskRadius < 0.625
                        pulseAnimation.start()
                        root._diskPulsating = true
                    }
                }
            }
        ]
    }

    Item {
        id: markItem

        width: shaderEffect.width
        height: shaderEffect.height
        visible: false
        layer.enabled: true
        layer.smooth: true

        Item {
            id: markIcon

            readonly property real lineWidth: width * Math.SQRT2 * 0.12
            readonly property real lineHypotenuse: lineWidth * Math.SQRT2 * 0.25
            readonly property real pathWidth: (leftSegment.width + rightSegment.width - markIcon.lineWidth) * progress
            readonly property bool checked: root.checked && !root.busy && shaderEffect.diskRadius === 1.0
            property real progress: 1.0

            anchors.centerIn: parent

            width: Theme.dp(18)
            height: width * 0.6 + lineHypotenuse * 2.0
            scale: 1.0 + (1.0 - progress) * 0.5
            opacity: 0.0

            onCheckedChanged: {
                opacityAnimation.stop()

                if (checked) {
                    markIcon.progress = 0.0
                    markIcon.opacity = 1.0
                    progressAnimation.start()
                } else {
                    opacityAnimation.start()
                }
            }

            NumberAnimation on progress {
                id: progressAnimation

                running: false
                duration: 350
                easing.type: Easing.InOutQuad
                to: 1.0
            }

            NumberAnimation on opacity {
                id: opacityAnimation

                running: false
                duration: 200
                easing.type: Easing.InOutQuad
                to: 0.0
            }

            Item {
                id: leftSegment

                readonly property real hypotenuse: markIcon.width * 0.2

                x: hypotenuse + markIcon.lineHypotenuse - width * 0.5
                y: markIcon.height - hypotenuse - markIcon.lineHypotenuse - height * 0.5
                width: hypotenuse * Math.SQRT2 * 2.0
                height: markIcon.lineWidth
                rotation: 45

                Rectangle {
                    width: Math.min(markIcon.pathWidth, parent.width)
                    height: parent.height
                    radius: height * 0.5
                }
            }

            Item {
                id: rightSegment

                readonly property real hypotenuse: markIcon.width * 0.3

                x: leftSegment.hypotenuse * 2.0 + hypotenuse - markIcon.lineHypotenuse - width * 0.5
                y: hypotenuse + markIcon.lineHypotenuse - height * 0.5
                width: hypotenuse * Math.SQRT2 * 2.0
                height: markIcon.lineWidth
                rotation: -45

                Rectangle {
                    width: Math.min(markIcon.pathWidth - leftSegment.width, parent.width)
                           + (markIcon.pathWidth > leftSegment.width ? markIcon.lineWidth : 0)
                    height: parent.height
                    radius: height * 0.5
                }
            }
        }
    }
}
