/**
 * SPDX-FileCopyrightText: 2023-2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0
import Nemo.FileManager 1.0
import Sailfish.FileManager 1.0
import Aurora.Controls 1.0

AppBar {
    id: root

    property var parentGridView
    property var popUpMenu
    property var popUpMenuSort
    property int errorType
    property string headerTitle
    property string pageTitle
    property string initialPath
    property string description
    property string path
    property int _headersHorizontalAlignment: Qt.AlignLeft
    property int countFiles: 0
    readonly property bool isUserDir: typeof path !== "undefined" ? FileEngine.isUserDir(path) : false
    readonly property bool isMaxButtonsVisible: isLandscape || isTablet
    property bool isSelectionMode
    property bool isPasteMode
    property bool isArchiveMode
    property bool isVirtualFolderMode
    property bool isSearchMode
    property bool isHomePageMode
    property bool isPinnedPanelMode
    property bool isSortingMode
    property bool isExtractMode
    property var standardPaths
    property var activeButtons: []
    property alias searchField: searchField
    readonly property bool isTablet: LayoutClassProvider.currentDisplayClass & DisplayLayoutClass.Tablet
    readonly property bool pickerMode: isPasteMode || isExtractMode
    readonly property bool noError: isArchiveMode ? errorType === ArchiveModel.NoError : errorType === FileModel.NoError
    readonly property bool hasFiles: countFiles > 0 && noError
    readonly property bool canShown: isTablet ? !orientationTransitionRunning : true
    readonly property var constActiveButtons: isMaxButtonsVisible
                                              ? [!isVirtualFolderMode ? searchButton.objectName
                                                                      : gridLayoutButton.objectName,
                                                 sortingButton.objectName, !isHomePageMode && !isVirtualFolderMode ? moreButton.objectName
                                                                                           : selectModeButton.objectName]
                                              : isSearchMode ? [selectModeButton.objectName, sortingButton.objectName]
                                                             : [!isVirtualFolderMode ? searchButton.objectName
                                                                                     : sortingButton.objectName,
                                                                moreButton.objectName]

    signal newDirectoryDialog()
    signal selectModeClicked()
    signal searchModeClicked()
    signal extractArchive()
    signal updateButtonVisibility()
    signal updateButtonVisibilityAfterRotation()
    signal updateButtonVisibilityFinished()
    signal selectAllClicked(bool selected)

    onConstActiveButtonsChanged: {
        if (!isTablet) {
            activeButtons = []
            updateButtonVisibilityAfterRotation()
        }
    }

    on_PageActiveChanged: {
        activeButtons = []
        updateButtonVisibilityAfterRotation()
    }

    onHasFilesChanged: {
        activeButtons = []
        updateButtonVisibilityAfterRotation()
    }

    onIsSelectionModeChanged: {
        if (isSelectionMode) {
            activeButtons = [selectedButton.objectName]
        }
        updateButtonVisibility()
    }

    function updateActiveButtons(objectName, buttonVisibility) {
        var activeButtons_ = []
        const isConstActive = constActiveButtons.indexOf(objectName) !== -1
        const isNotActive = activeButtons.indexOf(objectName) === -1

        if (buttonVisibility) {
            if (activeButtons.length < constActiveButtons.length && isConstActive) {
                if (isNotActive)
                    activeButtons.push(objectName)
            } else {
                if (isConstActive) {
                    for (var bt in activeButtons) {
                        if (constActiveButtons.indexOf(activeButtons[bt]) === -1
                                && isNotActive && (hasFiles || isVirtualFolderMode || isHomePageMode)) {
                            activeButtons.splice(bt, 1);
                            break
                        }
                    }
                    if (isNotActive)
                        activeButtons.push(objectName)
                } else {
                    activeButtons_.push(objectName)
                }
            }
        }

        for (var id in activeButtons_) {
            if (activeButtons.length < constActiveButtons.length) {
                if (activeButtons.indexOf(activeButtons_[id]) === -1)
                    activeButtons.push(activeButtons_[id])
            } else {
                break
            }
        }

        if (activeButtons.length > 0)
            updateButtonVisibility()
    }

    function checkButtonVisibility(objectName) {
        if (activeButtons.length > 0) {
            return activeButtons.indexOf(objectName) !== -1
        } else {
            return false
        }
    }

    function updateButtonVisible(buttonId, needUpdateFinished) {
        updateActiveButtons(buttonId.objectName, buttonId.buttonVisibility)
        buttonId.visible = buttonId.buttonVisibility && checkButtonVisibility(buttonId.objectName)

        updateButtonVisibility.connect(function () {
            buttonId.visible = buttonId.buttonVisibility && checkButtonVisibility(buttonId.objectName)

            if (needUpdateFinished)
                updateButtonVisibilityFinished()
        })

        updateButtonVisibilityAfterRotation.connect(function () {
            updateActiveButtons(buttonId.objectName, buttonId.buttonVisibility)

            if (needUpdateFinished)
                updateButtonVisibilityFinished()
        })
    }

    headerText: parentGridView.selectionMode ?
                    //% "%n object(s) selected"
                    qsTrId("files-la-selected_count", parentGridView.model.selectedCount) :
                    parentGridView.pasteMode ?
                        //% "Paste %n object(s)"
                        qsTrId("files-la-paste_count", FileEngine.clipboardCount) :
                        parentGridView.extractMode ?
                            //% "Extract to..."
                            qsTrId("filemanager-me-extract_to") :
                            parentGridView.model.path === initialPath && pageTitle.length > 0 ?
                                pageTitle : headerTitle
    subHeaderText: description

    AppBarSearchField {
        id: searchField

        visible: root.isSearchMode

        backgroundEnabled: true
    }

    AppBarSpacer {}

    AppBarButton {
        id: searchButton

        readonly property bool buttonVisibility: (!isSelectionMode && hasFiles && !pickerMode
                                                 && !isArchiveMode && !isVirtualFolderMode && !isSearchMode)
                                                 || (!isSelectionMode && isHomePageMode && canShown
                                                     && !isVirtualFolderMode && !pickerMode)
                                                 || isPinnedPanelMode

        objectName: "SearchButton"
        icon.source: "image://theme/icon-splus-search"
        visible: false

        onButtonVisibilityChanged: updateButtonVisible(searchButton)
        onClicked: searchModeClicked()
    }

    AppBarButton {
        id: sortingButton

        readonly property bool buttonVisibility: !isSelectionMode && hasFiles && !pickerMode
                                                 && noError && !isArchiveMode && !isHomePageMode
                                                 && canShown

        objectName: "SortingButton"
        icon.source: "image://theme/icon-m-sortby"
        visible: false

        onButtonVisibilityChanged: updateButtonVisible(sortingButton, true)
        onClicked: {
            isSortingMode = true
            popUpMenuSort.open(moreButton.visible ? moreButton : sortingButton)
        }
    }

    AppBarButton {
        id: gridLayoutButton

        readonly property bool buttonVisibility: ((!isSelectionMode && !pickerMode && !isSearchMode && hasFiles
                                                 && !isArchiveMode && sortingButton.visible)
                                                 || (isHomePageMode && countFiles !== 0 && !isSelectionMode && !pickerMode)
                                                 || (isVirtualFolderMode && hasFiles && !isSelectionMode && !pickerMode))
                                                 && canShown

        objectName: "GridLayoutButton"
        icon.source: FileManager.gridLayoutSetting.value ? "image://theme/icon-splus-list"
                                                         : "image://theme/icon-splus-grid"
        visible: false

        onButtonVisibilityChanged: updateButtonVisible(gridLayoutButton)
        onClicked: FileManager.gridLayoutSetting.value = !FileManager.gridLayoutSetting.value
    }

    AppBarButton {
        id: showHiddenButton

        readonly property bool buttonVisibility: isUserDir && noError && !isArchiveMode
                                                 && !isSelectionMode && !pickerMode && !isSearchMode && canShown

        objectName: "ShowHiddenButton"
        icon.source: FileManager.hiddenItemsSetting.value ? "image://theme/icon-splus-file-hide"
                                                          : "image://theme/icon-splus-file-show"
        visible: false

        onButtonVisibilityChanged: updateButtonVisible(showHiddenButton, true)
        onClicked: FileManager.hiddenItemsSetting.value = !FileManager.hiddenItemsSetting.value
    }

    AppBarButton {
        id: newFolderButton

        readonly property bool buttonVisibility: isUserDir && noError && showHiddenButton.visible && !isSearchMode
                                                 && !isArchiveMode && !isSelectionMode && !pickerMode && canShown

        objectName: "NewFolderButton"
        icon.source: "image://theme/icon-splus-new-folder"
        visible: false

        onButtonVisibilityChanged: updateButtonVisible(newFolderButton, true)
        onClicked: newDirectoryDialog()
    }

    AppBarButton {
        id: selectModeButton

        readonly property bool buttonVisibility: (isVirtualFolderMode || isHomePageMode || isSearchMode) && hasFiles
                                                 && !isSelectionMode && !pickerMode && canShown

        objectName: "SelectModeButton"
        icon.source: "image://theme/icon-splus-multiselect"
        visible: false

        onButtonVisibilityChanged: updateButtonVisible(selectModeButton, true)
        onClicked: selectModeClicked()
    }

    AppBarButton {
        id: selectedButton

        readonly property bool buttonVisibility: isSelectionMode && !pickerMode && hasFiles
                                                 && noError && canShown

        objectName: "SelectedAllButton"
        icon.source: "image://theme/icon-splus-multiselect"

        visible: buttonVisibility
        opacity: visible ? 1 : 0
        enabled: parentGridView.model.selectedCount !== parentGridView.model.count

        onButtonVisibilityChanged: updateActiveButtons(objectName, buttonVisibility)

        onClicked: {
            parentGridView.model.selectAllFiles()
            selectAllClicked(true)
        }

        Behavior on opacity { FadeAnimator {} }
    }

    AppBarButton {
        id: deselectButton

        readonly property bool buttonVisibility: selectedButton.buttonVisibility

        objectName: "DeselectAllButton"
        icon.source: "image://theme/icon-splus-multiclear"

        visible: buttonVisibility
        opacity: visible ? 1 : 0
        enabled: parentGridView.model.selectedCount !== 0

        onButtonVisibilityChanged: updateActiveButtons(objectName, buttonVisibility)

        onClicked: {
            parentGridView.model.clearSelectedFiles()
            selectAllClicked(false)
        }
    }

    AppBarButton {
        id: extractAllButton

        readonly property bool buttonVisibility: isArchiveMode && hasFiles && noError
                                                 && !pickerMode && canShown

        objectName: "ExtractAllButton"
        icon.source: "image://theme/icon-splus-extract-in"
        visible: buttonVisibility
        opacity: visible ? 1 : 0

        onButtonVisibilityChanged: updateActiveButtons(objectName, buttonVisibility)
        onClicked: extractArchive()
        Behavior on opacity { FadeAnimator {} }
    }

    AppBarButton {
        id: moreButton

        readonly property bool buttonVisibility: {
            if (!isSelectionMode && !pickerMode && !isSearchMode
                    && !isPinnedPanelMode && !isArchiveMode && noError && canShown) {
                return true
            }

            if (isHomePageMode && isMaxButtonsVisible || isHomePageMode && countFiles === 0)
                return false

            if (countFiles === 0 && parentGridView.model.path === initialPath)
                return false

            if (isVirtualFolderMode && !hasFiles)
                return false

            return !isSelectionMode && !pickerMode && noError
                    && !isPinnedPanelMode && !isArchiveMode && !isSearchMode && canShown
        }

        objectName: "MoreButton"
        icon.source: "image://theme/icon-splus-more"
        visible: false

        onButtonVisibilityChanged: updateButtonVisible(moreButton)
        onClicked: {
            isSortingMode = false
            popUpMenu.appBar = root
            popUpMenu.open(moreButton)
        }
    }

    BusyIndicator {
        id: busyIndicator

        objectName: "BusyIndicator"
        size: BusyIndicatorSize.ExtraSmall
        visible: running
        running: FileEngine.mode === FileEngine.CopyMode
                 || FileEngine.mode === FileEngine.MoveMode
                 || FileEngine.mode === FileEngine.DeleteMode
    }
}
