/*
 * SPDX-FileCopyrightText: Copyright 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.6
import ru.auroraos.WebView.Private 1.0
import ru.auroraos.WebView.Global 1.0
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Aurora.Controls 1.0

TextSelection {
    id: textSelection

    property var handles: ({})
    property var menu: menuItem
    property string handlePath: 'SelectionHandle.qml'
    property bool isShown: false

    function getHandle(id) {
        if (!(id in handles)) {
            var component = Qt.createComponent(handlePath)
            var handleObject = component.createObject(textSelection)
            handles[id] = handleObject
        }

        return handles[id]
    }

    onShowHandle: {
        var targetHandle = getHandle(handleId)
        if (targetHandle)
            targetHandle.show()
    }

    onHideHandle: {
        var targetHandle = getHandle(handleId)
        if (targetHandle)
            targetHandle.hide()
    }

    onHandleOpacityUpdated: {
        var targetHandle = getHandle(handleId)
        if (targetHandle)
            targetHandle.setOpacity(opacity)
    }

    onHandleOrientationUpdated: {
        var targetHandle = getHandle(handleId)
        if (targetHandle) {
            targetHandle.setOrientation(orientation)
        }
    }

    onMoveHandle: {
        var targetHandle = getHandle(handleId)
        if (targetHandle)
            targetHandle.move(location.x, location.y)
    }

    onShowMenu: {
        if (menu)
            menu.show(textLocation.x, textLocation.y, availableActions)
    }

    onHideMenu: {
        if (menu && isShown)
            menu.hide()
    }

    onExecuteCustomCommand: {
        console.log("EXECUTE", commandId)
    }

    onShareSelectedText: {
        shareDialog.resources = [{
                                     "name": "sharedText.txt",
                                     "data": text,
                                     "type": "text/plain",
                                     "status": text,
                                 }]
        shareDialog.trigger()
    }

    anchors.fill: parent
    handleSize: Qt.size(Theme.itemSizeExtraSmall, Theme.itemSizeExtraSmall)

    ShareAction {
        id: shareDialog

        title: qsTrId("webview-pi-share")
    }

    Item {
        id: topAnchor
    }


    Item {
        id: menuItem

        property int position: positionZero
        readonly property int positionZero: 0
        readonly property int positionLeft: 1
        readonly property int positionRight: 2

        function getLeftHandle() {
            for (var key in handles) {
                if (handles[key].isLeft() && handles[key].opacity > 0) {
                    return handles[key]
                }
            }
        }

        function getRightHandle() {
            for (var key in handles) {
                if (handles[key].isRight() && handles[key].opacity > 0) {
                    return handles[key]
                }
            }
        }

        function getCenterHandle() {
            for (var key in handles) {
                if (handles[key].isCenter() && handles[key].opacity > 0) {
                    return handles[key]
                }
            }
        }

        function show(newX, newY, availableActions) {
            var toIgnore = true
            for (var key in handles) {
                if (handles[key].opacity > 0) {
                    toIgnore = false
                }
            }

            if (toIgnore || getCenterHandle() &&
                    !(availableActions & QuickMenuEditStateFlags.QM_EDITFLAG_CAN_PASTE)) {
                isShown = false
                return
            }

            topAnchor.y = newY

            copyButton.visible = (availableActions & QuickMenuEditStateFlags.QM_EDITFLAG_CAN_COPY)
            cutButton.visible = (availableActions & QuickMenuEditStateFlags.QM_EDITFLAG_CAN_CUT)
            pasteButton.visible = (availableActions & QuickMenuEditStateFlags.QM_EDITFLAG_CAN_PASTE)
            shareButton.visible = !(getCenterHandle())
            searchButton.visible = !(getCenterHandle())

            contextMenu.open()
            isShown = true
        }

        function hide() {
            contextMenu.close()
            position = positionZero
            isShown = false
        }

        anchors {
            top: topAnchor.top
            topMargin: -Theme.paddingSmall
            bottom: getCenterHandle() ? getCenterHandle().bottom : position == positionLeft ?
                                            getLeftHandle().bottom :
                                            getRightHandle().bottom
        }
        x: {
            var leftHandle = getLeftHandle()
            var rightHandle = getRightHandle()
            var centerHandle = getCenterHandle()
            Math.min(Math.max((getCenterHandle() ? centerHandle.x + centerHandle.width / 2 : position == (positionRight) ?
                                                       rightHandle.x + rightHandle.width - contextMenu.preferredWidth / 2 :
                                                       position == (positionLeft) ?
                                                           leftHandle.x + contextMenu.preferredWidth / 2 :
                                                           (leftHandle.x + rightHandle.x + rightHandle.width) / 2),
                              contextMenu.preferredWidth / 2 + Theme.paddingMedium),
                     textSelection.width - contextMenu.preferredWidth / 2
                     - Theme.paddingLarge)
        }

        //special label, because labels in PopupItem don't know about Labels' width
        Label {
            id: popupLabelWidth

            visible: false
            text: copyButton.text + "\n" + shareButton.text + "\n" + searchButton.text + "\n"
                  + pasteButton.text + "\n" + cutButton.text
        }

        PopupMenu {
            id: contextMenu

            preferredHeight: Theme.dp(72) * 6
            preferredWidth: popupLabelWidth.width + Theme.paddingSmall * 8

            PopupMenuItem {
                id: copyButton

                onClicked: {
                    textSelection.execute(MenuId.MENU_ID_COPY)
                }

                //% "Copy"
                text: qsTrId("webview-pi-copy")
            }

            PopupMenuItem {
                id: shareButton


                onClicked: {
                    textSelection.execute(MenuId.MENU_ID_USER_SHARE)
                }

                //% "Share"
                text: qsTrId("webview-pi-share")
            }

            PopupMenuItem {
                id: searchButton

                onClicked: {
                    textSelection.execute(MenuId.MENU_ID_SEARCH)
                }

                //% "Search"
                text: qsTrId("webview-pi-search")
                enabled: false
            }

            PopupMenuItem {
                id: pasteButton

                onClicked: {
                    textSelection.execute(MenuId.MENU_ID_PASTE)
                }

                //% "Paste"
                text: qsTrId("webview-pi-paste")
            }

            PopupMenuItem {
                id: cutButton

                onClicked: {
                    textSelection.execute(MenuId.MENU_ID_CUT)
                }

                //% "Cut"
                text: qsTrId("webview-pi-cut")
            }

            // disable InverseMouseArea in Popup.qml for manual control of closing
            Component.onCompleted: contentItem.children[0].enabled = false
        }
    }
}
