/****************************************************************************************
**
** SPDX-FileCopyrightText: 2022-2024 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: BSD 3-Clause
**
****************************************************************************************/

import QtQuick 2.2
import Sailfish.Silica 1.0

Item {
    id: root

    anchors.left: internal.currentItem ? internal.currentItem.left : undefined
    opacity: internal.active ? 1.0 : 0.0

    property real progress

    signal clicked
    signal anchorChanged

    Behavior on opacity {
        FadeAnimation {
            id: opacityAnimation
            onRunningChanged: {
                if (!running) {
                    root.anchorTo(internal.pendingItem)
                }
            }
        }
    }

    function anchorTo(item, immediate) {
        internal.pendingItem = item
        if ((opacityAnimation.running || internal.active) && !immediate) {
            internal.active = item === internal.currentItem
        } else {
            internal.currentItem = item
            root.anchorChanged()
            internal.active = item !== null
        }
    }

    QtObject {
        id: internal
        property Item pendingItem
        property Item currentItem
        property bool active
    }

    SafeZoneTracker {
        width: Theme.pageStackIndicatorWidth
        height: Theme._itemSizeHeader

        PageStackChevronIndicator {
            id: indicator

            property alias backIndicatorActive: mouseArea.containsMouse
            readonly property bool dragGestureActive: root.progress > 0.0

            function updateIndicator() {
                if (backIndicatorActive) {
                    progressBehavior.enabled = true
                    progress = 1.0
                    return
                }

                if (dragGestureActive) {
                    progressBehavior.enabled = false
                    progress = Qt.binding(function() { return root.progress })
                    return
                }

                progressBehavior.enabled = true
                progress = 0.0
            }

            onBackIndicatorActiveChanged: updateIndicator()
            onDragGestureActiveChanged: updateIndicator()

            anchors.centerIn: parent
            anchors.horizontalCenterOffset: Theme.dp(3)
            compact: true

            Behavior on progress {
                id: progressBehavior

                NumberAnimation {
                    duration: 200
                    easing.type: Easing.InOutQuart
                }
            }
        }

        MouseArea {
            id: mouseArea

            onClicked: root.clicked()

            anchors.fill: parent

            enabled: internal.active && internal.currentItem != null
        }
    }
}
