/**
 * SPDX-FileCopyrightText: 2022-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef SAILFISH_MDM_SYSTEMUPDATE_H
#define SAILFISH_MDM_SYSTEMUPDATE_H

#include <QDateTime>
#include <QObject>
#include <QtGlobal>
#include <QVersionNumber>

namespace Sailfish {
namespace Mdm {

class SystemUpdatePrivate;
class Q_DECL_EXPORT SystemUpdate : public QObject
{
    Q_OBJECT

public:
    enum class Error {
        // Error mappings from the SystemUpdateManager namespace
        NoError = 0,
        IOError,
        NetworkError,
        MetadataError,
        PackageError,
        DBusError,
        SsuError,
        NotSupportedError,
        InternalError,
        BusyError,
        AuthenticationError,
        LowDisk,
        LowBattery,
        VerifyError,
        ConnectionError,
        WlanConditionError,
        RoamingConditionError,
        // Additional errors for MDM client
        NotAvailable = 100,
        ScheduleError,
        ScheduledInstallationError,

        UnknownError = -1,
    };
    Q_ENUM(Error)

    enum class Method {
        Any = 0,
        PackageUpdate = 1,
        ResetUpdate = 2,
        Unknown = -1
    };
    Q_ENUM(Method)

    struct ScheduledDownloadInfo {
        QString from = "";
        QString to = "";
        QString single_from = "";
        QString single_to = "";
        QString daily_from = "";
        QString daily_to = "";
        QString vDataUrl = "";
        QString version = "";
        Method method = Method::Any;
        int downloadProgress = 0; // percentage 0-100, not bytes count
        bool isDownloadingNow = false;
    };
    
    struct VersionEntry
    {
        QVersionNumber version;
        Method method;
    };

    Q_PROPERTY(bool ready READ ready NOTIFY readyChanged)
    Q_PROPERTY(bool scheduled READ scheduled NOTIFY scheduledChanged)
    Q_PROPERTY(int progress READ progress NOTIFY progressChanged)
    Q_PROPERTY(int downloadProgress READ downloadProgress NOTIFY downloadProgressChanged)
    Q_PROPERTY(int isDownloaded READ isDownloaded NOTIFY isDownloadedChanged)
    Q_PROPERTY(QString version READ version NOTIFY versionChanged)
    Q_PROPERTY(Method method READ method NOTIFY methodChanged)
    Q_PROPERTY(qlonglong size READ size NOTIFY sizeChanged)
    Q_PROPERTY(QDateTime lastChecked READ lastChecked NOTIFY lastCheckedChanged)

    explicit SystemUpdate(QObject *parent = nullptr);

    QString version() const;
    Method method() const;
    qlonglong size() const;
    QDateTime lastChecked() const;

    bool ready() const;
    bool scheduled() const;
    bool scheduled(QString &schedule);
    void cancelScheduled() const;
    Q_DECL_DEPRECATED int progress() const;
    int downloadProgress() const;
    void requestAvailableVersions();

    bool isDownloaded() const;
    bool isVendorDataDownloaded() const;
    void cancelScheduledDownload();
    struct ScheduledDownloadInfo isDownloadScheduled();

public slots:
    void downloadUpdate(const QString &version, Method method, const QString &vendorDataUrl = {});
    void scheduleDownload(const QDateTime from, const QDateTime to, const Method method,
        const QString &version, const QString &vendorDataUrl);
    void scheduleDownload(const QTime from, const QTime to, const Method method,
        const QString &version, const QString &vendorDataUrl);

    Q_DECL_DEPRECATED void scheduleUpdate(QDateTime from, QDateTime to);
    void scheduleUpdate(QDateTime from, QDateTime to, bool allowRecharge);
    void scheduleUpdate(QTime from, QTime to, bool allowRecharge);
    void onScheduledInstallationError(const QString &details);

    void installUpdate();
    void cancelDownload();
    void cancelUpdate();
    void clearDownloadedData();

signals:
    void availableVersionsReady(const QVector<VersionEntry> listOfVersionsAvailable);
    void isDownloadedChanged();
    void readyChanged();
    void scheduledChanged();
    void scheduledDownloadChanged();
    Q_DECL_DEPRECATED void progressChanged();
    void downloadProgressChanged();
    void versionChanged();
    void methodChanged();
    void sizeChanged();
    void lastCheckedChanged();
    void error(Error error, const QString &details);

private:
    SystemUpdatePrivate *d_ptr;
    Q_DISABLE_COPY(SystemUpdate)
    Q_DECLARE_PRIVATE(SystemUpdate)
};

}
}

#endif // SAILFISH_MDM_SYSTEMUPDATE_H
