/*
 * SPDX-FileCopyrightText: Copyright 2019-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 *
 * Copyright (c) 2018 – 2019 Jolla Ltd.
 *
 * License: Proprietary
 *
*/

import QtQuick 2.6
import QtGraphicalEffects 1.0
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0
import Nemo.Thumbnailer 1.0

GridItem {
    id: root

    readonly property var internalStorageRegExp: new RegExp("^" + StandardPaths.home)
    readonly property bool isInternalStorage: internalStorageRegExp.test(model.fileLocation)
    property string fileName: model.fileName
    property bool trackerMode
    property var mimeTypeContainer: !trackerMode ? [] : mimeTypeCallback()
    property string mimeType: !trackerMode ? model.mimeType : mimeTypeContainer["mimeType"]
    property double size: model.size
    property bool isDir: model.isDir
    property var created: model.created
    property var modified: model.modified
    property int nestedFiles: typeof model.nestedFiles !== "undefined" ? model.nestedFiles : 0
    property bool compressed
    property bool isArchive: !trackerMode ? FileEngine.isArchive(model.absolutePath) : mimeTypeContainer["isArchive"]
    property string searchText
    property var mimeTypeCallback

    readonly property bool isPicture: FileManager.isPicture(mimeType)
    readonly property bool isVideo: FileManager.isVideo(mimeType)
    readonly property var modelData: model

    property bool showStorageIcon

    contentWidth: gridView.gridLayout ? gridView.cellWidth - gridView.chunkWidth - Theme.paddingMedium : gridView.cellWidth - 2 * Theme.paddingMedium
    contentHeight: gridView.gridLayout ? gridView.cellHeight - Theme.paddingMedium : gridView.cellHeight - Theme.paddingSmall
    contentItem.radius: Theme.paddingSmall
    contentItem.anchors.centerIn: root
    opacity: gridView.pasteMode && !FileEngine.clipboardContainsCopy && down ? Theme.opacityLow : 1.0

    Loader {
        anchors {
            top: parent.top
            right: parent.right
            rightMargin: Theme.paddingMedium / 2
        }

        sourceComponent: storageIconComponent
        active: gridView.gridLayout && showStorageIcon
    }

    Grid {
        id: contentGrid

        anchors.fill: root.contentItem
        width: root.contentWidth
        height: root.contentHeight

        leftPadding: gridView.gridLayout ? Theme.paddingSmall : Theme.paddingMedium
        rightPadding: leftPadding
        topPadding: gridView.gridLayout ? Theme.paddingMedium / 3 * 4 : Theme.paddingSmall
        bottomPadding: gridView.gridLayout ? Theme.paddingMedium : topPadding

        rows: gridView.gridLayout ? 2 : 1
        columns: gridView.gridLayout ? 1 : 2

        horizontalItemAlignment: gridView.gridLayout ? Qt.AlignHCenter : Qt.AlignLeft
        verticalItemAlignment: gridView.gridLayout ? Qt.AlignTop : Qt.AlignVCenter

        spacing: gridView.gridLayout ? Theme.paddingSmall : Theme.paddingMedium

        Item {
            width: Theme.iconSizeLauncher
            height: width

            Rectangle {
                id: mask

                anchors.fill: parent
                radius: Theme.paddingSmall
                visible: false
            }

            Item {
                id: imageBackground

                anchors.fill: parent
                visible: false

                Rectangle {
                    anchors.fill: parent
                    rotation: gridView.gridLayout ? 0 : 270
                    gradient: Gradient {
                        // Abusing gradient for inactive mimeTypes
                        GradientStop { position: 0.0; color: Theme.rgba(Theme.primaryColor, 0.1) }
                        GradientStop { position: 1.0; color: "transparent" }
                    }
                }

                Loader {
                    id: loader

                    anchors.centerIn: parent
                    sourceComponent: isPicture || isVideo ? pictureComponent : iconComponent
                }

                Component {
                    id: pictureComponent

                    Item {
                        width: imageBackground.width
                        height: imageBackground.height

                        Thumbnail {
                            objectName: "PictireThumbnail"

                            anchors.fill: parent
                            mimeType: root.mimeType
                            priority: Thumbnail.NormalPriority
                            source: fileName
                            sourceSize.width: parent.width
                            sourceSize.height: parent.height
                        }

                        HighlightImage {
                            objectName: "PictureHighlightImage"

                            anchors.fill: parent
                            visible: root.isVideo
                            cache: true
                            source: root.isDir
                                    ? "image://theme/icon-m-file-folder"
                                    : Theme.iconForMimeType(root.mimeType)
                        }
                    }
                }

                Component {
                    id: iconComponent

                    HighlightImage {
                        objectName: "IconHighlightImage"

                        anchors.centerIn: parent
                        cache: true
                        smooth: false
                        source: {
                            if (root.isDir)
                                return "image://theme/icon-m-file-folder"

                            if (root.isArchive)
                                return "image://theme/icon-m-file-archive-folder"

                            return Theme.iconForMimeType(root.mimeType)
                        }
                    }
                }

                Image {
                    objectName: "CompressedImage"

                    anchors {
                        top: parent.top
                        right: parent.right
                    }

                    cache: true
                    smooth: false
                    visible: compressed
                    source: {
                        var iconSource = "image://theme/icon-s-file-compressed-displaced"
                        return iconSource + (highlighted ? "?" + Theme.highlightColor : "")
                    }
                }
            }

            OpacityMask {
                anchors.fill: parent
                source: imageBackground
                maskSource: mask
            }
        }

        Row {
            width: gridView.gridLayout
                   ? contentGrid.width - Theme.paddingSmall * 2
                   : contentGrid.width - contentGrid.spacing - imageBackground.width
            height: column.height

            Column {
                id: column

                width: parent.width - storageIcon.rightMargin
                spacing: Theme.paddingSmall
                bottomPadding: gridView.gridLayout
                               ? nameLabel.lineCount > 1 ? 0 : nameMetrics.height
                : 0

                Label {
                    id: nameLabel
                    objectName: "NameLabel"

                    width: gridView.gridLayout ? parent.width : parent.width - Theme.horizontalPageMargin
                    text: {
                        var _fileName = root.fileName.indexOf("/") >= 0
                                ? root.fileName.substring(root.fileName.lastIndexOf("/") + 1, root.fileName.length)
                                : root.fileName

                        return root.searchText.length > 0 ? Theme.highlightText(_fileName, root.searchText.replace("*", ""), Theme.highlightColor)
                                                          : _fileName
                    }

                    elide: gridView.gridLayout ? Text.ElideNone : Text.ElideMiddle
                    wrapMode: gridView.gridLayout ? Text.WrapAnywhere : Text.NoWrap
                    maximumLineCount: gridView.gridLayout ? 2 : 1
                    horizontalAlignment: gridView.gridLayout ? Qt.AlignHCenter : Qt.AlignLeft
                    font.pixelSize: gridView.gridLayout ? Theme.fontSizeExtraSmall : Theme.fontSizeMedium
                    color: dateLabel.highlighted ? Theme.highlightColor : Theme.primaryColor
                }

                Label {
                    id: sizeLabel
                    objectName: "SizeLabel"

                    width: nameLabel.width
                    text: root.isDir
                    //% "%n object(s)"
                          ? qsTrId("filemanager-la-nested_objects", root.nestedFiles)
                          : Format.formatFileSize(root.size)

                    color: highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                    horizontalAlignment: gridView.gridLayout ? Qt.AlignHCenter : Qt.AlignRight
                    font.pixelSize: gridView.gridLayout ? Theme.fontSizeTiny : Theme.fontSizeExtraSmall

                    Label {
                        id: dateLabel
                        objectName: "DateLabel"

                        width: parent.width
                        text: Qt.formatDateTime(root.modified || root.created, "d MMMM yyyy hh:mm")
                        color: highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                        horizontalAlignment: Qt.AlignLeft
                        font.pixelSize: gridView.gridLayout ? Theme.fontSizeTiny : Theme.fontSizeExtraSmall
                        visible: !gridView.gridLayout
                    }

                    Loader {
                        id: storageIcon

                        readonly property int rightMargin: active ? Theme.horizontalPageMargin : 0

                        anchors {
                            left: dateLabel.right
                            verticalCenter: dateLabel.verticalCenter
                        }

                        sourceComponent: storageIconComponent
                        active: !gridView.gridLayout && showStorageIcon
                    }
                }
            }
        }

        Component {
            id: storageIconComponent

            StorageIcon {
                isInternal: root.isInternalStorage
            }
        }

        FontMetrics {
            id: nameMetrics
            font.pixelSize: dateLabel.font.pixelSize
        }
    }
}
