/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Copyright (c) 2021 Open Mobile Platform LLC.
 */

"use strict";

var gInputMethodHandler = null;

function debug(msg) {
  Logger.debug("InputMethodHandler.js -", msg);
}

XPCOMUtils.defineLazyModuleGetters(this, {
  Services: "resource://gre/modules/Services.jsm",
});

/**
  * InputMethodHandler
  *
  * Provides surrounding text, cursor and anchor position of editable
  * element for predictive input.
  */
InputMethodHandler.prototype = {
  QueryInterface: ChromeUtils.generateQI([Ci.nsIObserver,
                                          Ci.nsISupportsWeakReference,
                                          Ci.nsISelectionListener]),

  // Weak-ref used to keep track of the currently focused element.
  _currentFocusedElement: null,

  // Save context to not resend it several times
  _inputContext: {surroundingText: null, cursorPosition: 0, anchorPosition: 0},

  _init: function() {
    Logger.debug("JSScript: InputMethodHandler.js loaded");
    addEventListener("focus", this, true);
    addEventListener("blur", this, true);
    addEventListener("input", this, false);
  },

  get focusedElement() {
    return this._currentFocusedElement && this._currentFocusedElement.get();
  },

  notifySelectionChanged: function(doc, sel, reason) {
    this._sendInputContext(this.focusedElement);
  },

  handleEvent: function(aEvent) {
    switch (aEvent.type) {
      case "focus": {
        let currentElement = aEvent.target;
        if (this._isTextInput(currentElement)) {
          this._currentFocusedElement = Cu.getWeakReference(currentElement);
          this._sendInputAttributes(currentElement);
          if (this._isInputContextAvailable(currentElement)) {
            this._startSelectionMonitor(currentElement);
            this._sendInputContext(currentElement);
          }
        }

        break;
      }

      case "blur": {
        let focused = this.focusedElement;
        if (focused) {
          this._resetInputAttributes(focused);
          if (this._isInputContextAvailable(focused)) {
            this._stopSelectionMonitor(focused);
            this._resetInputContext(focused);
          }
        }

        this._currentFocusedElement = null;
        break;
      }

      case "selectionchange": {
        let focused = this.focusedElement;
        let selectionIn = aEvent.target.getSelection().anchorNode.parentElement;

        if (focused.contains(selectionIn))
          this._sendInputContext(focused);

        break;
      }

      case "input":
        this._sendInputContext(aEvent.target);
        break;
    }
  },

  _updateInputContext: function(aElement) {
    let surroundingText = null;
    let cursorPosition = 0;
    let anchorPosition = 0;

    if ('selectionStart' in aElement && 'selectionEnd' in aElement && 'value' in aElement) {
      surroundingText = aElement.value;
      cursorPosition = aElement.selectionStart;
      anchorPosition = aElement.selectionEnd;
    } else {
      let selection = aElement.ownerDocument.getSelection();
      if (selection.rangeCount) {
        surroundingText = selection.focusNode.textContent;
        let range = selection.getRangeAt(0);
        cursorPosition = range.startOffset;
        anchorPosition = range.endOffset;
      }
    }

    if (this._inputContext.surroundingText === surroundingText
        && this._inputContext.cursorPosition === cursorPosition
        && this._inputContext.anchorPosition === anchorPosition) {
      return false
    }

    this._inputContext.surroundingText = surroundingText;
    this._inputContext.cursorPosition = cursorPosition;
    this._inputContext.anchorPosition = anchorPosition;

    return true;
  },

  _sendInputContext: function(aElement) {
    if (!this._isInputContextAvailable(aElement) || aElement !== this.focusedElement) {
      return;
    }

    if (!this._updateInputContext(aElement)) {
      return
    }

    try {
      let winId = Services.embedlite.getIDByWindow(aElement.ownerGlobal);
      Services.embedlite.sendAsyncMessage(winId, "InputMethodHandler:SetInputContext",
                                          JSON.stringify({surroundingText: this._inputContext.surroundingText,
                                                          cursorPosition: this._inputContext.cursorPosition,
                                                          anchorPosition: this._inputContext.anchorPosition}));
    } catch (e) {
      Logger.warn("InputMethodHandler: sending async message failed", e);
    }
  },

  _resetInputContext: function(aElement) {
    try {
      let winId = Services.embedlite.getIDByWindow(aElement.ownerGlobal);
      Services.embedlite.sendAsyncMessage(winId, "InputMethodHandler:ResetInputContext", "[]");
    } catch (e) {
      Logger.warn("InputMethodHandler: sending async message failed", e);
    }

    this._inputContext.surroundingText = null;
    this._inputContext.cursorPosition = 0;
    this._inputContext.anchorPosition = 0;
  },

  _sendInputAttributes: function(aElement) {
    try {
      let winId = Services.embedlite.getIDByWindow(aElement.ownerGlobal);
      Services.embedlite.sendAsyncMessage(winId, "InputMethodHandler:SetInputAttributes",
                                          JSON.stringify({autocomplete: aElement.getAttribute("autocomplete") || "on",
                                                          autocapitalize: aElement.getAttribute("autocapitalize") || "on"}));
    } catch (e) {
      Logger.warn("InputMethodHandler: sending async message failed", e);
    }
  },

  _resetInputAttributes: function(aElement) {
    try {
      let winId = Services.embedlite.getIDByWindow(aElement.ownerGlobal);
      Services.embedlite.sendAsyncMessage(winId, "InputMethodHandler:ResetInputAttributes", "[]");
    } catch (e) {
      Logger.warn("InputMethodHandler: sending async message failed", e);
    }
  },

  _isTextInput: function(aElement) {
    return Util.isEditable(aElement) &&
           !this._isDisabledElement(aElement) &&
           ('readOnly' in aElement ? !aElement.readOnly : true)
  },

  _isInputContextAvailable: function(aElement) {
    return this._isTextInput(aElement) &&
           (aElement.type !== "password");
  },

  _isDisabledElement: function(aElement) {
    let currentElement = aElement;
    while (currentElement) {
      if (currentElement.disabled) {
        return true;
      }
      currentElement = currentElement.parentElement;
    }
    return false;
  },

  _startSelectionMonitor: function(aElement) {
    try {
      if (aElement.editor) {
        let selection = aElement.editor.selectionController
                                .getSelection(Ci.nsISelectionController.SELECTION_NORMAL);
        selection.addSelectionListener(this);
      } else
        aElement.ownerDocument.addEventListener("selectionchange", this);
    } catch (e) {
      Logger.warn("InputMethodHandler: adding selection listener failed", e);
    }
  },

  _stopSelectionMonitor: function(aElement) {
    try {
      if (aElement.editor) {
        let selection = aElement.editor.selectionController
                                .getSelection(Ci.nsISelectionController.SELECTION_NORMAL);
        selection.removeSelectionListener(this);
      } else
        aElement.ownerDocument.removeEventListener("selectionchange", this);
    } catch (e) {
      Logger.warn("InputMethodHandler: removing selection listener failed", e);
    }
  },

};

function InputMethodHandler() {
  this._init();
}

gInputMethodHandler = new InputMethodHandler();
