/*
 * SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

#ifndef MDMAPPLICATIONS_H
#define MDMAPPLICATIONS_H

#include <QList>
#include <QObject>
#include <QString>
#include <QtGlobal>

namespace Sailfish {
namespace Mdm {
class ApplicationEntryPrivate;
class PackageEntryPrivate;
class ApplicationOperationPrivate;
class ApplicationsPrivate;

class Q_DECL_EXPORT PackageEntry
{
public:
    PackageEntry();
    PackageEntry(const PackageEntry &other);
    Q_DECL_DEPRECATED PackageEntry(const QString &name,
                                   const QString &version,
                                   const QString &vendor);
    PackageEntry(const QString &name,
                 const QString &version,
                 const QString &vendor,
                 const QString &release);
    PackageEntry &operator=(const PackageEntry &other);

    virtual ~PackageEntry();

    bool isValid() const;
    QString name() const;
    QString version() const;
    QString vendor() const;
    QString release() const;

private:
    PackageEntryPrivate *d_ptr;
};

class Q_DECL_EXPORT ApplicationEntry
{
public:
    ApplicationEntry(const QString &name, const PackageEntry &package);
    ApplicationEntry(const QString &name,
                     const PackageEntry &package,
                     const QStringList &permissions);
    ApplicationEntry(const ApplicationEntry &other);
    ApplicationEntry &operator=(const ApplicationEntry &other);
    virtual ~ApplicationEntry();

    QString name() const;
    QStringList permissions() const;
    PackageEntry package() const;
    QString iconPath() const;

private:
    ApplicationEntryPrivate *d_ptr;
    Q_DECLARE_PRIVATE(ApplicationEntry)
};

class Q_DECL_EXPORT ApplicationOperation : public QObject
{
    Q_OBJECT

public:
    bool isFinished() const;
    bool isSucceeded() const;

    Q_DECL_DEPRECATED QString package() const;
    Q_DECL_DEPRECATED QString version() const;
    QStringList packages() const;
    QStringList failedPackages() const;

    QString errorString() const;

signals:
    void finished(ApplicationOperation *applicationOperation);
    void failed(ApplicationOperation *applicationOperation);
    void succeeded(ApplicationOperation *applicationOperation);

private slots:
    void setError(const QString &errorString);
    void setFinished();
    void setFinishedWithError();

protected:
    explicit ApplicationOperation();
    virtual ~ApplicationOperation();

private:
    friend class Applications;
    ApplicationOperationPrivate *d_ptr;
    Q_DISABLE_COPY(ApplicationOperation)
    Q_DECLARE_PRIVATE(ApplicationOperation)
};

class Q_DECL_EXPORT Applications : public QObject
{
    Q_OBJECT

public:
    enum PackageKitOperation { InstallFromRepos, InstallFromFiles, Remove };
    enum PackageFilter { Installed, Newest, None };

    explicit Applications(QObject *parent = 0);
    virtual ~Applications();

    QList<ApplicationEntry> getInstalledApplications();
    QList<ApplicationEntry> getInstalledApplications(uint uid);
    ApplicationEntry getInstalledApplicationInfo(uint uid = 0, QString appName = "");
    QList<PackageEntry> getInstalledPackages();
    bool grantPermissions(uint uid, bool always, const QString &application);

    ApplicationOperation *installPackage(const QString &package, const QString &version = QString());
    ApplicationOperation *installPackages(const QStringList &packages,
                                          const QMap<QString, QString> &versions);
    ApplicationOperation *removePackage(const QString &package, bool autoremove);

private:
    void processPackagesByName(const QStringList &packages,
                               const QMap<QString, QString> &versions,
                               PackageKitOperation pkOperation,
                               bool autoremove = false);
    void installOrRemoveFromRepo(const PackageKitOperation pkOperation,
                                 const QStringList pkgIds,
                                 bool autoremove);
    void resolvePackages(const QStringList &packages,
                         const QMap<QString, QString> &versions,
                         PackageFilter packageFilter,
                         PackageKitOperation pkOperation,
                         bool autoremove);
    void installPackagesFromFiles(const QStringList packages);
    void installPackagesViaAPM(ApplicationOperation *operation, const QStringList packages);
    void logAborted(PackageKitOperation pkOperation, const QString &str);
    ApplicationsPrivate *d_ptr;
    Q_DISABLE_COPY(Applications)
    Q_DECLARE_PRIVATE(Applications)

signals:
    void operationAborted(const PackageKitOperation op, const QString error);
    void operationSucceeded(const PackageKitOperation op);
};

} // namespace Mdm
} // namespace Sailfish

#endif // MDMAPPLICATIONS_H