/****************************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2020-2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
*
** Copyright (c) 2013-2020. Jolla Ltd.
** All rights reserved.
**
** This file is part of Sailfish Silica UI component package.
**
** You may use this file under the terms of BSD license as follows:
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are met:
**     * Redistributions of source code must retain the above copyright
**       notice, this list of conditions and the following disclaimer.
**     * Redistributions in binary form must reproduce the above copyright
**       notice, this list of conditions and the following disclaimer in the
**       documentation and/or other materials provided with the distribution.
**     * Neither the name of the Jolla Ltd nor the
**       names of its contributors may be used to endorse or promote products
**       derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
** DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
** ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
****************************************************************************************/

/****************************************************************************
**
** Copyright (C) 2011 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the Qt Components project.
**
** $QT_BEGIN_LICENSE:BSD$
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
** $QT_END_LICENSE$
**
****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private

Private.SilicaMouseArea {
    id: page

    // The status of the page. One of the following:
    //      PageStatus.Inactive - the page is not visible
    //      PageStatus.Activating - the page is transitioning into becoming the active page
    //      PageStatus.Active - the page is the current active page
    //      PageStatus.Deactivating - the page is transitioning into becoming inactive
    readonly property int status: __stack_container
                                  ? __stack_container.status
                                  : (parent && parent.hasOwnProperty("__auroracontrols_splitview") && visible ? PageStatus.Active : PageStatus.Inactive)

    property bool backNavigation: true
    property bool showNavigationIndicator: true
    property bool forwardNavigation: _belowTop
    property bool canNavigateForward: forwardNavigation
    property int navigationStyle: PageNavigation.Horizontal
    readonly property bool _horizontalNavigationStyle: navigationStyle === PageNavigation.Horizontal
    property Item pageContainer
    property Item __stack_container

    property color backgroundColor: Theme.rgba(palette.overlayBackgroundColor, 0)
    property Component background: {
        if (backgroundColor.a !== 0) {
            return backgroundComponent
        } else if (pageContainer) {
            return pageContainer.pageBackground
        } else {
            return null
        }
    }

    property bool highContrast

    property int allowedOrientations: __silica_applicationwindow_instance._defaultPageOrientations
    property int orientation: orientationState.orientation

    property alias orientationTransitions: orientationState.transitions
    property alias defaultOrientationTransition: orientationState.defaultTransition
    property bool orientationTransitionRunning

    property bool isPortrait: (orientation === Orientation.Portrait || orientation === Orientation.PortraitInverted || orientation === Orientation.None)
    property bool isLandscape: (orientation === Orientation.Landscape || orientation === Orientation.LandscapeInverted)

    readonly property int _navigation: __stack_container ? __stack_container.navigation : PageNavigation.NoNavigation
    readonly property int _navigationPending: __stack_container ? __stack_container.navigationPending : PageNavigation.NoNavigation
    readonly property int _direction: __stack_container ? __stack_container.direction : PageNavigation.NoDirection

    property int _allowedOrientations: {
        var allowed = allowedOrientations & __silica_applicationwindow_instance.allowedOrientations
        if (!allowed) {
            // No common supported orientations, let the page decide
            allowed = allowedOrientations
        }
        return allowed
    }
    property alias _windowOrientation: orientationState.pageOrientation

    property int _horizontalDimension: parent ? parent.width : Screen.width
    property int _verticalDimension: parent ? parent.height : Screen.height

    property var _forwardDestination
    property var _forwardDestinationProperties
    property int _forwardDestinationAction: PageStackAction.Push
    property Item _forwardDestinationInstance
    property var _forwardDestinationReplaceTarget
    property int _depth: parent && parent.hasOwnProperty("pageStackIndex") ? parent.pageStackIndex : -1

    property alias _windowOpacity: page.opacity

    property bool _opaqueBackground: background !== null && background != backgroundComponent
    readonly property bool _exposed:  pageContainer
                && __stack_container
                && pageContainer.visible
                && ((pageContainer._currentContainer === __stack_container)
                    || (pageContainer._currentContainer !== null && (pageContainer._currentContainer === __stack_container.transitionPartner)))
    readonly property bool _belowTop: pageContainer
                && __stack_container
                && pageContainer._currentContainer === __stack_container
                && __stack_container.attachedContainer !== null
    property bool _clickablePageIndicators: true

    // The pending orientation state is used to avoid early changes the page geometry,
    // before recalculating all possible page margins, paddings or insets.
    property bool _orientationPending

    property int __silica_page
    readonly property Item __silica_page_instance: page

    function _orientationChangeDefaultAction() {
        // All orientation-related properties have been changed to their
        // relevant values, use default action for updating the SafeZoneRect
        // immediately and changing page geometry with new dimensions.
        SafeZoneRect.updateImmediately()
        _orientationPending = false
    }

    onOrientationChanged: {
        if (orientationState.defaultTransitionRunning) {
            return
        }
        if (orientationState.pageOrientation === orientation) {
            // Reset pending state if used no default transitions.
            _orientationPending = false
        }
    }

    onOrientationTransitionRunningChanged: {
        if (!orientationTransitionRunning) {
            // Reset pending state anyway when no transitions running.
            _orientationPending = false
        }
    }

    // This unusual binding avoids a warning when the page is destroyed.
    anchors.centerIn: page ? parent : null

    width: orientationState.width
    height: orientationState.height

    opacity: orientationTransitionRunning, 1
    scale: orientationTransitionRunning, 1

    rotation: orientationState.rotation
    visible: __stack_container && __stack_container.visible
    focus: true

    Component {
        id: backgroundComponent

        Rectangle {
            property color backgroundColor: page.backgroundColor

            onBackgroundColorChanged: {
                if (backgroundColor.a !== 0) {
                    color = backgroundColor
                }
            }
        }
    }

    StateGroup {
        id: dimensionState

        states: [
            State {
                when: !page.pageContainer || !page._exposed || !orientationState.completed
                      || page._orientationPending
                      || orientationState.pageOrientation === Orientation.None
                      || orientationState.pageOrientation !== page.orientation
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    // Unbind previous values and hold them until the page
                    // is not exposed or not rotated to the desired orientation.
                    width: orientationState.width
                    height: orientationState.height
                }
            },
            State {
                name: 'PortraitMask'
                when: orientationState.pageOrientation & Orientation.PortraitMask
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    width: page._horizontalDimension
                    height: page._verticalDimension
                }
            },
            State {
                name: 'LandscapeMask'
                when: orientationState.pageOrientation & Orientation.LandscapeMask
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    width: page._verticalDimension
                    height: page._horizontalDimension
                }
            }
        ]
    }

    StateGroup {
        id: orientationState

        property bool completed
        property bool defaultTransitionRunning
        // Choose the orientation this page will have given the current device orientation
        property int pageOrientation: Orientation.None
        property int desiredPageOrientation: pageContainer && pageContainer.hasOwnProperty("__silica_window_pagestack")
                                             ? __silica_applicationwindow_instance._selectOrientation(page._allowedOrientations, __silica_applicationwindow_instance.deviceOrientation)
                                             : Orientation.Portrait
        property bool desiredPageOrientationSuitable: pageContainer && pageContainer.hasOwnProperty("__silica_window_pagestack")
                                                      ? desiredPageOrientation & __silica_applicationwindow_instance.deviceOrientation
                                                      : true

        // These are the state managed orientation derived properties. Normally the pages properties
        // of the same name are bound to these but just prior to the orientation state changing those
        // bindings will be broken so property changes due to state fast-forwarding aren't propagated.
        property real width: page.isPortrait ? page._horizontalDimension : page._verticalDimension
        property real height: page.isPortrait ? page._verticalDimension : page._horizontalDimension
        property real orientation:  Orientation.Portrait
        property real rotation

        onDesiredPageOrientationChanged: _updatePageOrientation()
        onDesiredPageOrientationSuitableChanged: _updatePageOrientation()

        function _updatePageOrientation() {
            if (pageOrientation !== desiredPageOrientation) {
                // Lock in the current values of the orientation properties, when the pageOrientation
                // property and the states which derive from it change the fast-forwarded values
                // will be the same as the
                page.orientation = page.orientation
                page.rotation = page.rotation
                page.width = page.width
                page.height = page.height
                page._orientationPending = true

                const init = pageOrientation === Orientation.None
                pageOrientation = desiredPageOrientation

                // State fast forwarding is done, restore the bindings.
                page.orientation = Qt.binding(function() { return orientationState.orientation })
                page.rotation = Qt.binding(function() { return orientationState.rotation })
                page.width = Qt.binding(function() { return orientationState.width })
                page.height = Qt.binding(function() { return orientationState.height })

                if (init) {
                    _orientationChangeDefaultAction()
                }
            }
        }

        state: 'Unanimated'
        states: [
            State {
                name: 'Unanimated'
                when: !page.pageContainer || !page._exposed || !completed
                PropertyChanges {
                    target: page
                    restoreEntryValues: false
                    // If a transition is interrupted part way through reset page constants a
                    // transition may have animated to fixed values and not finalised.
                    orientationTransitionRunning: false
                    _orientationPending: false
                }
            },
            State {
                name: 'Portrait'
                when: orientationState.pageOrientation === Orientation.Portrait ||
                      orientationState.pageOrientation ===  Orientation.None
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    rotation: 0
                    orientation: Orientation.Portrait
                }
            },
            State {
                name: 'Landscape'
                when: orientationState.pageOrientation === Orientation.Landscape
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    rotation: 90
                    orientation: Orientation.Landscape
                }
            },
            State {
                name: 'PortraitInverted'
                when: orientationState.pageOrientation === Orientation.PortraitInverted
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    rotation: 180
                    orientation: Orientation.PortraitInverted
                }
            },
            State {
                name: 'LandscapeInverted'
                when: orientationState.pageOrientation === Orientation.LandscapeInverted
                PropertyChanges {
                    target: orientationState
                    restoreEntryValues: false
                    rotation: 270
                    orientation: Orientation.LandscapeInverted
                }
            }
        ]

        property Transition defaultTransition: Private.PageOrientationTransition {
            onRunningChanged: orientationState.defaultTransitionRunning = running

            targetPage: page
            orientationChangeActions: [
                ScriptAction {
                    // All orientation-related properties have been changed to their
                    // relevant values, use default action for updating the SafeZoneRect
                    // immediately and changing page geometry with new dimensions.
                    script: page._orientationChangeDefaultAction()
                }
            ]
        }

        Component.onCompleted: {
            if (transitions.length === 0) {
                transitions = [ defaultTransition ]
            }
        }
    }

    Component.onCompleted: {
        orientationState.completed = true
        orientation = orientationState.pageOrientation
    }
}
