/**
 * SPDX-FileCopyrightText: Copyright 2023-2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.2
import Sailfish.Silica 1.0
import Sailfish.FileManager 1.0
import Nemo.FileManager 1.0
import Aurora.Controls 1.0

PopupMenu {
    id: root

    readonly property int verticalPadding: 3 * Theme.paddingSmall
    property string mimeType
    property bool extended: root.isSelectOneItem
    property bool singleItemSelected
    property bool virtualFolder
    property bool searchPage
    property bool homePage
    property bool pageHeaderMode
    property bool showNewFolder
    property bool isUserDir
    property bool hasFiles
    property bool isSelectionMode
    property bool isSortMode
    property bool isSelectOneItem
    property bool isArchive
    property bool isGalleryPreviewMode
    property bool standardPath
    property bool isDir
    property var appBar

    readonly property InverseMouseArea inverseMouseArea: {
        for (var i = 0; i < root.contentItem.children.length; ++i) {
            const child = root.contentItem.children[i]

            if (child.pressedOutside) {
                return child
            }
        }

        return null
    }

    signal deleteClicked
    signal shareClicked
    signal detailsClicked
    signal renameClicked
    signal copyClicked
    signal cutClicked
    signal locationClicked
    signal compressClicked
    signal newDirectoryClicked
    signal deleteDirectoryClicked
    signal sortClicked(int sortOrder, int sortBy, int directorySort)
    signal selectModeClicked
    signal extractClicked(bool needSelection)
    signal pasteModeClicked
    signal updateVisible
    signal openUrlExternallyClicked
    signal pressedOutside

    preferredHeight: Theme.dp(72) * root.items.length
    preferredWidth: Screen.sizeCategory > Screen.Medium
                    ?  Math.max(Screen.width, Screen.height) * 0.33333333 - Theme.horizontalPageMargin
                    :  Math.min(Screen.width, Screen.height) - Theme.horizontalPageMargin * 2

    PopupMenuItem {
        onClicked: root.deleteClicked()

        objectName: "DeleteItem"
        //% "Delete"
        text: qsTrId("filemanager-me-delete")
        visible: root.extended && !deleteFolderMenuItem.visible && !standardPath
    }

    PopupMenuItem {
        onClicked: root.shareClicked()

        objectName: "ShareItem"
        //% "Share"
        text: qsTrId("filemanager-me-share")
        visible: root.extended && !root.isDir
    }

    PopupMenuItem {
        onClicked: root.copyClicked()

        objectName: "CopyItem"
        //% "Copy"
        text: qsTrId("filemanager-me-copy")
        visible: (!root.pageHeaderMode && root.extended) || root.isGalleryPreviewMode
    }

    PopupMenuItem {
        onClicked: root.cutClicked()

        objectName: "CutItem"
        visible: (!virtualFolder || searchPage) && !standardPath && !pageHeaderMode
        //% "Cut"
        text: qsTrId("filemanager-me-cut")
    }

    PopupMenuItem {
        onClicked: root.detailsClicked()

        objectName: "DetailsItem"
        //% "Details"
        text: qsTrId("filemanager-me-details")
        visible: root.singleItemSelected
    }

    PopupMenuItem {
        onClicked: root.renameClicked()

        objectName: "RenameItem"
        //% "Rename"
        text: qsTrId("filemanager-me-rename")
        visible: root.singleItemSelected && !standardPath
    }

    PopupMenuItem {
        onClicked: root.compressClicked()

        objectName: "AddToArchiveItem"
        visible: !pageHeaderMode
        //% "Add to archive"
        text: qsTrId("filemanager-me-add_to_archive")
    }

    PopupMenuItem {
        onClicked: root.locationClicked()

        objectName: "ObjectLocationItem"
        visible: searchPage && singleItemSelected
        //% "Object location"
        text: qsTrId("filemanager-me-object_location")
    }

    PopupSubMenuItem {
        id: submenuSort

        function checkVisibility() {
            visible = hasFiles
                    && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("SortingButton") === -1)
                    && !homePage && (!isSelectOneItem || !isSelectionMode) && pageHeaderMode
        }

        objectName: "SortItem"
        visible: {
            checkVisibility()
            updateVisible.connect(function () {
                checkVisibility()
            })
        }

        text: qsTrId("filemanager-he-sort")

        PopupMenuSortItems {
            onSortClicked: root.sortClicked(sortOrder, sortBy, directorySort)
        }
    }


    PopupMenuItem {
        function checkVisibility() {
            visible = pageHeaderMode
                    && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("GridLayoutButton") === -1)
                    && hasFiles && !isSelectOneItem && !isSelectionMode
        }

        onClicked: FileManager.gridLayoutSetting.value = !FileManager.gridLayoutSetting.value

        objectName: "ListDisplayItem"
        visible:  {
            checkVisibility()
            updateVisible.connect(function () {
                checkVisibility()
            })
        }

        //% "List display"
        text: FileManager.gridLayoutSetting.value ? qsTrId("filemanager-me-list_view")
                                                    //% "Grid display"
                                                  : qsTrId("filemanager-me-grid_view")
    }

    PopupMenuItem {
        //% "Show"
        readonly property string showHidden: qsTrId("filemanager-me-dont_show_hidden_item")
        //% "Don`t show"
        readonly property string dontShowHidden: qsTrId("filemanager-me-show_hidden_item")

        function checkVisibility() {
            visible = isUserDir
                    && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("ShowHiddenButton") === -1)
                    && (!isSelectOneItem || !isSelectionMode) && pageHeaderMode
        }

        onClicked: {
            root.close()
            FileManager.hiddenItemsSetting.value = !FileManager.hiddenItemsSetting.value
        }
        objectName: "HiddenitemsItem"
        visible: {
            checkVisibility()
            updateVisible.connect(function () {
                checkVisibility()
            })
        }
        //% "%1 hidden items"
        text: qsTrId("filemanager-me-hidden_items").arg(FileManager.hiddenItemsSetting.value ? dontShowHidden : showHidden)
    }

    PopupMenuItem {
        function checkVisibility() {
            visible = !virtualFolder && pageHeaderMode && showNewFolder && isUserDir
                    && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("NewFolderButton") === -1)
        }

        onClicked: root.newDirectoryClicked()

        objectName: "NewFolderItem"
        height: Theme.itemSizeSmall
        visible: {
            checkVisibility()
            updateVisible.connect(function () {
                checkVisibility()
            })
        }

        //% "New folder"
        text: qsTrId("filemanager-me-new_folder")
    }

    PopupMenuItem {
        id: deleteFolderMenuItem

        onClicked: root.deleteDirectoryClicked()

        objectName: "DeleteFolderItem"
        height: Theme.itemSizeSmall
        visible: !virtualFolder && pageHeaderMode && page.showDeleteFolder && isUserDir
                 && page.path !== page.initialPath && standardPaths.indexOf(fileModel.path) == -1

        //% "Delete folder"
        text: qsTrId("filemanager-me-delete_folder")
    }

    PopupMenuItem {
        function getVisibility() {
            return (pageHeaderMode
                    && (showNewFolder || virtualFolder)
                    && hasFiles
                    && (typeof(appBar) === "undefined" || appBar.activeButtons.indexOf("SelectModeButton") === -1))
                    || isSelectOneItem
        }

        onClicked: root.selectModeClicked()

        objectName: "SelectMultipleItem"
        height: Theme.itemSizeSmall
        visible: getVisibility()

        //% "Select multiple"
        text: qsTrId("filemanager-me-select_multiple")

        Component.onCompleted: {
            updateVisible.connect(function () { this.visible = Qt.binding(getVisibility) })
        }
    }

    PopupMenuItem {
        onClicked: root.extractClicked(false)

        objectName: "ExtractHereItem"
        height: Theme.itemSizeSmall
        visible: isArchive && singleItemSelected && !virtualFolder && !searchPage

        //% "Extract here"
        text: qsTrId("filemanager-me-extract_here")
    }

    PopupMenuItem {
        onClicked: root.extractClicked(true)

        objectName: "ExtractToItem"
        height: Theme.itemSizeSmall
        visible: isArchive && singleItemSelected

        //% "Extract to..."
        text: qsTrId("filemanager-me-extract_to")
    }

    PopupMenuItem {
        onClicked: root.pasteModeClicked()

        objectName: "PasteFilesItem"
        height: Theme.itemSizeSmall
        visible: pageHeaderMode && isUserDir && FileEngine.clipboardCount > 0
        text: !virtualFolder && !searchPage ? page._getPasteTextId() : ""
    }

    PopupMenuItem {
        readonly property string currentApp: {
            if (FileManager.isPicture(root.mimeType) || FileManager.isVideo(root.mimeType))
                //% "gallery"
                return qsTrId("filemanager-me-open_in_gallery")

            return ""
        }

        onClicked: root.openUrlExternallyClicked()

        objectName: "OpenInItem"
        visible: isGalleryPreviewMode
        //% "Open in %1"
        text: qsTrId("filemanager-me-open_in").arg(currentApp)
    }


    PopupMenuItem {
        // Service element
        visible: false
        Connections {
            target: FileManager
            onUpdatePopUpContent: updateVisible()
        }

        Connections {
            target: root.inverseMouseArea

            onPressedOutside: root.pressedOutside()
        }

        Timer {
            interval: 100
            running: true
            onTriggered: updateVisible()
        }
    }
}

