/**
 * SPDX-FileCopyrightText: Copyright 2022 – 2023 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
 */

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Pickers 1.0
import Sailfish.FileManager 1.0 as FileManager
import Nemo.FileManager 1.0

FileManager.RenameDialog {
    id: dialog

    property bool needSelectDirectory
    property string archiveExtension
    property string currentDirectoryPath
    property int archiveType
    property ExtractorView extractorView
    property var files
    property var __directory_page: _currentPageStack.find(function(dialog) { return dialog.hasOwnProperty("__directory_page") })
    property bool hasEncryption
    readonly property PageStack _currentPageStack: !homePage._isTablet ? pageStack : secondPanelPageStack
    readonly property bool correctPassword: archiveSettingsLoader.item.passwordActivitySwitch.correctMatch
                                            || !archiveSettingsLoader.item.passwordActivitySwitch.checked

    property string passwordErrorLabel
    readonly property bool isValidPath: currentDirectoryPath !== StandardPaths.home && currentDirectoryPath.length > 0

    objectName: "CreateArchiveDialog"

    onAccepted: {
        if (!passwordDialogMode) {
            extractorView.compressFiles(files, nameField.text, directoryPath, archiveType, archiveSettingsLoader.item.password)
        } else {
            extractorView.passwordNeeded = false
            extractorView.model.doActionWithPassword(archiveSettingsLoader.item.password)
        }

        FileManager.FileManager.selectedFilesCleared()

        if (__home_page !== null) {
            __home_page.dialogPageStack.clear()
        }
    }

    title: ""
    archiveCreationMode: true
    directoryPath: isValidPath ? currentDirectoryPath : ""

    acceptDestination: __directory_page
    acceptDestinationAction: PageStackAction.Pop
    canAccept: _newPath !== "" && !_exists && !_hasFirstSpace && !_hasEmptyName && !_hasFirstDot && directoryPath !== "" && correctPassword
    archiveSettingsLoader.sourceComponent: archiveSettings

    ArchiveTypeModel {
        id: archiveTypeModel
    }

    Component {
        id: archiveSettings

        Column {
            property alias passwordActivitySwitch: passwordSwitch
            readonly property string password: generalPasswordField.text

            width: parent.width
            spacing: Theme.paddingLarge

            ArchiveComboBoxItem {
                //% "Archive format"
                label: qsTrId("filemanager-la-archive_format")
                visible: !dialog.passwordDialogMode
                menu: ContextMenu {
                    Repeater {
                        model: archiveTypeModel
                        MenuItem {
                            text: model.displayName
                            onClicked: {
                                var fileName = dialog.nameField.text.replace(dialog.archiveExtension, "")
                                dialog.archiveExtension = model.displayName
                                dialog.archiveType = model.type
                                fileName = fileName.concat(dialog.archiveExtension)
                                dialog.nameField.text = fileName
                                dialog.hasEncryption = model.hasEncryption
                            }
                        }
                    }
                }

                Component.onCompleted: {
                    dialog.archiveExtension = archiveTypeModel.displayName(currentIndex)
                    dialog.archiveType = archiveTypeModel.archiveType(currentIndex)
                    dialog.hasEncryption = archiveTypeModel.hasEncryption(currentIndex)
                    dialog.oldPath = dialog.files.length === 1 ? dialog.files[0] + dialog.archiveExtension
                                                               : dialog.currentDirectoryPath + "/" + dialog.fileNameWithoutExt + dialog.archiveExtension
                }
            }

            ArchiveComboBoxItem {
                //% "Archive location"
                label: qsTrId("filemanager-la-archive_location")
                currentIndex: !needSelectDirectory ? 1 : 0
                visible: !dialog.passwordDialogMode
                value: {
                    if (needSelectDirectory || !isValidPath) {
                        //% "%1"
                        return qsTrId("filemanager-me-сreate_to").arg(dialog.directoryPath.split("/").pop())
                    } else {
                        //% "Current directory"
                        return qsTrId("filemanager-me-сurrent_directory")
                    }
                }

                //% "Archive will be created to %1"
                description: isValidPath || dialog.directoryPath.length > 0
                             ? qsTrId("filemanager-me-will_be_saved_to_download").arg(dialog.directoryPath)
                             : qsTrId("filemanager-me-select_archive_archive_folder")
                menu: ContextMenu {
                    MenuItem {
                        //% "Select a archive folder"
                        text: qsTrId("filemanager-me-select_archive_archive_folder")
                        onClicked: pageStack.animatorPush(folderPickerPage)
                    }

                    MenuItem {
                        visible: isValidPath
                        text: qsTrId("filemanager-me-сurrent_directory")
                        onClicked: {
                            needSelectDirectory = false
                            directoryPath = currentDirectoryPath
                        }
                    }
                }
            }

            TextSwitch {
                id: passwordSwitch

                readonly property bool correctMatch: !generalPasswordField.errorHighlight
                                                     && secondPasswordField.text.length > 0
                                                     && enabled

                onEnabledChanged: if (!enabled) checked = false

                //% "Set password"
                text: qsTrId("filemanager-la-set_password")
                enabled: dialog.hasEncryption
                visible: !dialog.passwordDialogMode

                Component.onCompleted: dialog.flickable.interactive = isPortrait ? checked : true

                onCheckedChanged: {
                    dialog.flickable.interactive = isPortrait ? checked : true

                    if (checked) {
                        generalPasswordField.forceActiveFocus()
                        dialog.flickable.contentY = dialog.flickable.contentHeight - dialog.flickable.height + Theme.paddingLarge / 2
                    } else {
                        dialog.flickable.scrollToTop()
                    }
                }

                Behavior on opacity { FadeAnimation {} }
            }

            Item {
                width: parent.width
                height: Theme.paddingMedium
                visible: dialog.passwordDialogMode
            }

            PasswordField {
                id: generalPasswordField

                property Item passwordVisibilityButton

                label: !errorHighlight || dialog.passwordDialogMode
                                         //% "Enter password"
                                       ? qsTrId("filemanager-la-enter_password")
                                         //% "Passwords mismatch"
                                       : qsTrId("filemanager-la-enter_password_match_error")
                description: errorHighlight ? passwordErrorLabel : ""
                opacity: passwordSwitch.checked || dialog.passwordDialogMode ? 1 : 0
                onTextChanged: {
                    errorHighlight = secondPasswordField.text != text && secondPasswordField.text.length > 0

                    if (dialog.passwordDialogMode) {
                        if (text.length > 0 && !errorHighlight) {
                            dialog.canAccept = true
                        } else {
                            dialog.canAccept = false

                        }
                    }
                }
                errorHighlight: passwordErrorLabel.length > 0
                states: [
                    State {
                        when: generalPasswordField.errorHighlight
                        PropertyChanges {
                            target: generalPasswordField
                            rightItem: passwordVisibilityButton
                        }
                    },
                    State {
                        when: !generalPasswordField.errorHighlight
                        PropertyChanges {
                            target: generalPasswordField
                            rightItem: passwordVisibilityButton
                        }
                    }
                ]

                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.onClicked: secondPasswordField.focus = true
                Component.onCompleted: {
                    if (dialog.passwordDialogMode) {
                        generalPasswordField.forceActiveFocus()
                    }

                    passwordVisibilityButton = passwordVisibilityComponent.createObject(generalPasswordField)
                }

                Component {
                    id: passwordVisibilityComponent

                    IconButton {
                        onClicked: {
                            if (generalPasswordField._automaticEchoModeToggle) {
                                generalPasswordField._usePasswordEchoMode = !generalPasswordField._usePasswordEchoMode
                            }
                            generalPasswordField._echoModeToggleClicked()
                        }

                        width: icon.width + 2 * Theme.paddingMedium
                        height: icon.height

                        enabled: generalPasswordField.showEchoModeToggle
                        opacity: generalPasswordField.showEchoModeToggle ? 1.0 : 0.0
                        icon.color: generalPasswordField.errorHighlight ? Theme.errorColor : Theme.primaryColor
                        icon.highlightColor: generalPasswordField.errorHighlight ? Theme.errorColor : Theme.highlightColor
                        icon.source: "image://theme/icon-splus-" + (generalPasswordField.echoMode === TextInput.Password ? "show-password"
                                                                                                                         : "hide-password")
                        Behavior on opacity { FadeAnimation {} }
                    }
                }

                Behavior on opacity { FadeAnimation {} }
            }

            PasswordField {
                id: secondPasswordField

                //% "Repeat password"
                label: qsTrId("filemanager-la-repeat_password")
                opacity: passwordSwitch.checked && !dialog.passwordDialogMode ? 1 : 0

                onTextChanged: generalPasswordField.errorHighlight = generalPasswordField.text != text && text.length > 0

                EnterKey.iconSource: "image://theme/icon-m-enter-accept"
                EnterKey.onClicked: if (dialog.canAccept) dialog.accept()
                Behavior on opacity { FadeAnimation {} }
            }
        }
    }

    Component {
        id: folderPickerPage

        FolderPickerPage {
            id: self

            showSystemFiles: false
            //% "Create to"
            dialogTitle: qsTrId("filemanager-ti-create-to")

            onSelectedPathChanged: {
                dialog.needSelectDirectory = true
                dialog.directoryPath = self.selectedPath
            }
        }
    }
}
