/*******************************************************************************
**
** SPDX-FileCopyrightText: 2023-2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: BSD-3-Clause
**
*******************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import "private/Util.js" as Util

SilicaItem {
    id: template

    property alias icon: iconImage
    property alias primaryText: primaryTextLabel.text
    property alias secondaryText: secondaryTextLabel.text
    property alias description: descriptionLabel.text

    property real _iconXOffset: Theme.paddingMedium
    property Cover _cover
    property int _coverOrientation: _cover
                                    ? _cover.orientation
                                    : width > height
                                        ? Cover.Horizontal
                                        : Cover.Vertical
    property real _coverActionAreaHeight: _cover
                                          ? _cover.coverActionArea.height
                                          : _isHorizontalCover
                                             ? Theme._coverActionsAreaHorizontalHeight
                                             : Theme._coverActionsAreaVerticalHeight
    readonly property bool _isHorizontalCover: _coverOrientation === Cover.Horizontal

    anchors.fill: parent

    Column {
        id: column

        x: Theme.paddingCover
        y: primaryItem.visible ? 0 : (_isHorizontalCover ? Theme.paddingMedium : Theme.paddingLarge)
        width: parent.width - x * 2

        Item {
            id: primaryItem

            width: parent.width
            height: primaryTextLabel.implicitHeight
            visible: primaryText.length > 0 || iconImage.status != Image.Null

            Label {
                id: primaryTextLabel

                width: Math.min(implicitWidth, parent.width - (iconImage.status === Image.Ready ? iconImage.width + Theme.paddingCover : 0))
                truncationMode: TruncationMode.Fade
                textFormat: Text.PlainText
                font.pixelSize: Theme.dp(60)
                visible: text.length > 0
            }

            Image {
                id: iconImage

                anchors {
                    verticalCenter: parent.verticalCenter
                    left: primaryTextLabel.right
                    leftMargin: primaryText.length ? Theme.paddingMedium : 0
                }

                width: Theme.iconSizeSmallPlus
                height: width
                fillMode: Image.PreserveAspectFit
                visible: status != Image.Null
            }
        }

        Label {
            id: secondaryTextLabel

            width: parent.width
            visible: secondaryText.length
            color: template.palette.primaryColor
            wrapMode: Text.WordWrap
            textFormat: Text.PlainText
            maximumLineCount: 2
            font.pixelSize: Theme.fontSizeExtraSmall
            font.weight: Font.Medium
        }
    }

    Label {
        id: descriptionLabel

        anchors {
            left: column.left
            right: column.right
            top: column.bottom
            topMargin: secondaryTextLabel.visible ? Theme.paddingSmall : 0
        }

        visible: y + height <= parent.height - _coverActionAreaHeight + Theme.paddingMedium
        color: template.palette.secondaryColor
        wrapMode: _isHorizontalCover ? Text.NoWrap : Text.WordWrap
        truncationMode: _isHorizontalCover ? TruncationMode.Fade : TruncationMode.None
        textFormat: Text.PlainText
        maximumLineCount: {
            if (_isHorizontalCover) {
                return secondaryText.length ? 2 - secondaryTextLabel.lineCount : 2
            }
            return secondaryText.length ? 6 - secondaryTextLabel.lineCount : 7
        }
        font.pixelSize: Theme.fontSizeTiny
    }

    Component.onCompleted: {
        _cover = Util.findParentWithProperty(template, "coverActionArea")
    }
}
