/****************************************************************************
**
** SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
****************************************************************************/

import QtQuick 2.5
import Aurora.Controls 1.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private

PickerDialog {
    id: root

    property var _contentModel
    property string _searchPlaceholderText
    property alias _emptyPlaceholderText: gridView.emptyPlaceholderText

    _clearOnBackstep: false

    orientationTransitions: Private.PageOrientationTransition {
        fadeTarget: _background ? gridView : __silica_applicationwindow_instance.contentItem
        targetPage: root
    }

    PickerGridView {
        id: gridView

        onMovementStarted: {
            searchField.focus = false
            searchItem.searchFieldFocus = false
        }

        anchors.topMargin: searchItem.height + searchItem.y + Theme.paddingMedium
        anchors.bottomMargin: toolBar.height + SafeZoneRect.insets.bottom

        isDialog: true
        itemModel: root._contentModel
        model: root._contentModel.model
        selectedCount: _selectedCount
        searchText: _contentModel.filter
    }

    SearchSlidingItem {
        id: searchItem

        contentModel: _contentModel
        appBarSearchField: searchField
        gridView: gridView
        placeholderText: root._searchPlaceholderText
    }

    PickerDialogHeader {
        id: dialogHeader

        readonly property bool searchFieldVisible: (SafeZoneRect._orientation & Orientation.LandscapeMask)
                                                   && dialogHeader.width > (4 * Theme.horizontalPageMargin
                                                                            + dialogTextMetrics.width
                                                                            + screen.height * 0.333)

        selectedCount: gridView.selectedCount
        showBack: !_clearOnBackstep
        divider: searchItem.y === SafeZoneRect.insets.top
        //% "Accept"
        acceptText: qsTrId("components_pickers-he-multipicker_attach-empty")
        extraContent.width: parent.width
        extraContent.x: 0
        // Search field must be above accept button
        extraContent.z: 1
        reserveExtraContent: false
        singleSelectionMode: _contentModel.singleSelectionMode

        extraContent.children: Item {
            function _updateComponentsWidth() {} // Mock appbar internal stuff

            anchors.verticalCenter: parent.verticalCenter
            height: parent.height
            width: parent.width

            AppBarSearchField {
                id: searchField

                onTextChanged: {
                    if (visible && _contentModel) {
                        _contentModel.filter = text.toLowerCase().trim()
                        searchItem.text = text
                    }
                }

                onVisibleChanged: {
                    searchItem.enableAnimation = false
                    if (visible) {
                        searchItem.visible = false
                    } else {
                        if (activeFocus) {
                            searchItem.forceActiveFocus()
                            searchItem.active = true
                        }
                        if (text != "") {
                            searchItem.active = true
                        }

                        searchItem.visible = true
                    }

                    searchItem.enableAnimation = true
                }

                EnterKey.onClicked: focus = false

                anchors.centerIn: parent

                visible: dialogHeader.searchFieldVisible
                placeholderText: root._searchPlaceholderText
                backgroundEnabled: true
                preferredWidth: Math.max(screen.height * 0.333, minimalWidth)

                EnterKey.iconSource: "image://theme/icon-m-keyboard-enter-close"
            }
        }

        TextMetrics {
            id: dialogTextMetrics

            text: dialogHeader.cancelText + dialogHeader.acceptText
            font {
                pixelSize: Theme.fontSizeMedium
                family: Theme.fontFamily
            }
        }
    }

    AppBar {
        id: toolBar

        readonly property bool longVersion: {
            return width > (2 * Theme.horizontalPageMargin // Spacing around toolBar content
                            + 2 * Theme.paddingSmall // Spacing between buttons
                            + Theme.dp(72 * 3) // Icon width
                            + 1.5 * Theme.horizontalPageMargin // Padding from side of the buttonText
                            + appBarTextMetrics.width // Text width
                            + (dialogHeader.searchFieldVisible ? 0: Theme.paddingSmall + Theme.dp(72)
                                                                    + 0.5 * Theme.horizontalPageMargin))
        }

        // Hide toolBar when keyboard is shown
        height: Math.max(Theme._itemSizeHeader - __silica_applicationwindow_instance.pageStack.panelSize, 0)
        _bottomPosition: true
        background: true
        divider: false

        AppBarButton {
            onClicked: {
                if (searchItem.active) {
                    searchItem.text = ""
                    searchItem.focus = false
                    searchItem.active = false
                } else {
                    searchItem.active = true
                    searchItem.forceActiveFocus()
                }
            }

            icon.source: "image://theme/icon-splus-search"
            //% "Search"
            text: qsTrId("components_pickers-la-search")
            highlighted: searchItem.active
            visible: !dialogHeader.searchFieldVisible
        }

        AppBarSpacer { }

        AppBarButton {
            onClicked: gridView.gridLayout = !gridView.gridLayout

            icon.source: "image://theme/icon-splus-%1".arg(gridView.gridLayout ? "list" : "cage")

            text: toolBar.longVersion ? (gridView.gridLayout
                                           //% "Show list"
                                         ? qsTrId("components_pickers-la-show_grid")
                                           //% "Show grid"
                                         : qsTrId("components_pickers-la-show_list"))
                                      : ""
        }

        AppBarSpacer {
            fixedWidth: dialogHeader.searchFieldVisible ? Theme.dp(64) : 0
            visible: toolBar.longVersion
        }

        AppBarButton {
            onClicked: {
                for (var i = 0; i < _contentModel.model.count; i++) {
                    if (_contentModel.model.get(i).selected) {
                        _contentModel.updateSelected(i, false)
                    }
                }
            }
            //% "Clear selection"
            text: toolBar.longVersion ? qsTrId("pickers-me-clear_selection") : ""
            icon.source: "image://theme/icon-splus-multiclean"
        }

        AppBarSpacer {
            fixedWidth: dialogHeader.searchFieldVisible ? Theme.dp(64) : 0
            visible: toolBar.longVersion
        }

        AppBarButton {
            onClicked: {
                for (var i = 0; i < _contentModel.model.count; i++) {
                    if (!_contentModel.model.get(i).selected) {
                        _contentModel.updateSelected(i, true)
                    }
                }
            }

            //% "Select all"
            text: toolBar.longVersion ? qsTrId("pickers-me-select_all") : ""
            icon.source: "image://theme/icon-splus-multiselect"
        }

        AppBarSpacer {
            visible: dialogHeader.searchFieldVisible
        }
    }

    TextMetrics {
        id: appBarTextMetrics

        //% "Clear selection"
        text: qsTrId("pickers-me-clear_selection")
                //% "Select all"
              + qsTrId("pickers-me-select_all")
                                           //% "Search"
              + (dialogHeader.searchFieldVisible ? "" : qsTrId("components_pickers-la-search"))
              + (gridView.gridLayout
                   //% "Show list"
                 ? qsTrId("components_pickers-la-show_grid")
                   //% "Show grid"
                 : qsTrId("components_pickers-la-show_list"))
        font {
            pixelSize: Theme.fontSizeSmall
            family: Theme.fontFamily
        }
    }
}
