/**
 * SPDX-FileCopyrightText: Copyright 2024 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: Proprietary
*/

import QtQuick 2.6
import Aurora.Controls 1.0

/*!
    \qmltype LazyPopupMenu
    \inherits LazyLoader
    \inqmlmodule Aurora.Applications.Internal
    \brief Lazy Aurora::PopupMenu.
*/
LazyLoader {
    id: root

    /*!
      Override instance with Aurora::PopupMenu type.
      */
    readonly property PopupMenu instance: root.item

    /*!
      Internal Aurora::PopupMenu::opened flag.
      */
    readonly property bool opened: root.instance
                                   ? root.instance.opened
                                   : false

    /*!
        Dont wait for Aurora::PopupMenu::progress before
        Aurora::PopupMenu::opened become true.
        Can be used to avoid delegate blinking.
       */
    readonly property bool opening: d.opening

    /*!
      Is internal Aurora::PopupMenu opened or in progress of opening.
      */
    readonly property bool openedOrInProgress: root.opened ||
                                               root.opening ||
                                               root.status === Loader.Loading

    /*!
      Open internal Aurora::PopupMenu.
      */
    function open() {
        root.promise({
            resolve: function(instance) {
                d.opening = true

                instance.open()
            }
        })
    }

    /*!
      Close internal Aurora::PopupMenu.
      */
    function close() {
        /* Dont call promise.
           There is not way we can close before open. */
        if (!root.instance) {
            console.warn("Cannot close. No instance")
            return
        }

        root.instance.close()
    }

    /* Fill parent so Aurora::PopupMenu can be positioned. */
    anchors.fill: parent

    onOpenedChanged: {
        if (root.opened) {
            /*
                If animation is over - root.opened become true.
                So we can drop d.opening flag.
            */
            d.opening = false
        }
    }

    QtObject {
        id: d

        property bool opening: false
    }
}
