/****************************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2023 Open Mobile Platform LLC <community@omp.ru>
**
** SPDX-License-Identifier: BSD 3-Clause
**
** All rights reserved.
**
** This file is part of Sailfish Silica UI component package.
**
** You may use this file under the terms of BSD license as follows:
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are met:
**     * Redistributions of source code must retain the above copyright
**       notice, this list of conditions and the following disclaimer.
**     * Redistributions in binary form must reproduce the above copyright
**       notice, this list of conditions and the following disclaimer in the
**       documentation and/or other materials provided with the distribution.
**     * Neither the name of the Jolla Ltd nor the
**       names of its contributors may be used to endorse or promote products
**       derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
** DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
** LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
** ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
****************************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0

Page {
    property QtObject controller
    property Page page

    allowedOrientations: page ? page.allowedOrientations : Orientation.All

    Component.onCompleted: {
        var menuIndex = 0
        var menuItems = controller.menu.children
        for (var i = 0; i < menuItems.length; i++) {
            var child = menuItems[i]
            if (child && child.hasOwnProperty("__silica_menuitem")) {
                if (child.visible) {
                    items.append( {"item": child, "menuIndex": menuIndex } )
                }
                menuIndex++
            }
        }
    }

    ListModel {
        id: items
    }

    SilicaListView {
        anchors.fill: parent
        model: items

        header: PageHeader {
            title: controller.comboBox.label
        }

        delegate: BackgroundItem {
            id: delegateItem

            readonly property bool _isMultiLine: descriptionLabel.text.length || mainLabel.lineCount > 1

            height: Math.max(labelColumn.height + Theme.paddingMedium*2, Theme.itemSizeSmall)

            onClicked: {
                model.item.clicked()
                controller.menu.activated(model.menuIndex)
                pageStack.pop()
            }

            HighlightImage {
                id: icon

                anchors {
                    left: parent.left
                    leftMargin: Theme.horizontalPageMargin

                    // If there is only one line of text, vertically center the icon.
                    // Otherwise, anchor it to the delegate's top, with a small padding.
                    top: _isMultiLine ? parent.top : undefined
                    topMargin: Theme.paddingMedium + Theme.paddingSmall
                    verticalCenter: _isMultiLine ? undefined : parent.verticalCenter
                }
                sourceSize.width: Theme.iconSizeMedium
                sourceSize.height: Theme.iconSizeMedium
                source: model.item.icon ? model.item.icon.source : ""
                highlighted: mainLabel.highlighted
                monochromeWeight: !!model.item.icon && model.item.icon.monochromeWeight !== undefined
                                  ? model.item.icon.monochromeWeight
                                  : 1.0
            }

            Column {
                id: labelColumn

                anchors {
                    left: model.item.icon ? icon.right : parent.left
                    leftMargin: model.item.icon ? Theme.paddingMedium : Theme.horizontalPageMargin
                    right: parent.right
                    rightMargin: Theme.horizontalPageMargin
                    verticalCenter: parent.verticalCenter
                }

                Label {
                    id: mainLabel

                    width: parent.width
                    wrapMode: Text.Wrap
                    text: model.item.text
                    font: model.item.font
                    highlighted: delegateItem.highlighted || model.item === controller.currentItem
                }

                Label {
                    id: descriptionLabel

                    width: parent.width
                    height: text.length ? implicitHeight : 0
                    font.pixelSize: Theme.fontSizeExtraSmall
                    wrapMode: Text.Wrap
                    text: model.item.description || ""
                    highlighted: mainLabel.highlighted
                    color: highlighted ? Theme.secondaryHighlightColor : Theme.secondaryColor
                }
            }
        }

        VerticalScrollDecorator {}
    }
}
