/****************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2019-2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
** Copyright (c) 2016 – 2019 Jolla Ltd.
**
** License: Proprietary
****************************************************************************/

import QtQuick 2.6
import Sailfish.Silica 1.0
import Sailfish.Share 1.0
import Nemo.FileManager 1.0
import Sailfish.FileManager 1.0
import org.nemomobile.configuration 1.0

Page {
    id: page

    property alias path: fileModel.path
    property alias header: gridView.header
    property string initialPath: homePath
    property string homePath // deprecated
    onHomePathChanged: if (homePath.length > 0) console.warn("DirectoryPage.homePath deprecated, use initialPath instead.")

    property string title
    property alias description: gridView.description
    property bool showFormat
    property var showDeleteFolder: page.path !== page.initialPath && standardPaths.indexOf(fileModel.path) == -1
    property bool mounting
    property bool showNewFolder: true
    property bool openSymLink: true
    property bool multiselection
    property alias sortBy: fileModel.sortBy
    property alias sortOrder: fileModel.sortOrder
    property alias caseSensitivity: fileModel.caseSensitivity
    property alias directorySort: fileModel.directorySort
    property alias errorType: fileModel.errorType

    property int __directory_page
    property string _deletingPath
    readonly property string __directory_path: path

    readonly property bool indexIfExists: true
    readonly property var standardPaths: [
        StandardPaths.documents,
        StandardPaths.download,
        StandardPaths.pictures,
        StandardPaths.music,
        StandardPaths.videos,
        StandardPaths.home + "/Playlists",
        StandardPaths.home + "/Templates",
        StandardPaths.home + "/Public",
        StandardPaths.home + "/Desktop"
    ]
    /*!
    \internal

    Implementation detail for file manager
    */
    property FileManagerNotification errorNotification
    property ExtractorView extractor

    signal formatClicked

    signal folderDeleted(string path)

    function refresh() {
        fileModel.refresh()
    }

    function _commitDeletion() {
        FileEngine.deleteFiles(_deletingPath, true)
    }

    function _getPasteTextId() {
        var path = FileEngine.clipboardFiles[0]

        if (FileEngine.clipboardCount > 1) {
            //: Shown when inserting a group of files, e.g. "Paste all"
            //% "Paste all"
            return qsTrId("filemanager-me-paste_all")
        } else if (path) {
            //: Shows the filename when paste, e.g. "Paste myFile.txt"
            //% "Paste %1"
            return qsTrId("filemanager-me-paste-with-arg").arg(path.split("/").pop())
        } else {
            //% "Paste"
            return qsTrId("filemanager-me-paste")
        }
    }

    function openNewDirectoryDialog() {
        gridView.openThirdColumn(Qt.resolvedUrl("NewFolderDialog.qml"), {
                                     path: page.path
                                 })
    }

    backNavigation: !FileEngine.busy

    Component.onCompleted: {
        if (!errorNotification) {
            errorNotification = errorNotificationComponent.createObject(page)
        }

        FileManager.init(pageStack)
        if (!FileManager.errorNotification) {
            FileManager.errorNotification = errorNotification
        }

        FileOperationMonitor.instance()
        SortingSettings.setSortType(sortBy, sortOrder, directorySort)

        directoryPathModel.path = typeof page !== "undefined" && page.path.length > 0 ? page.path : ""

        if (extractor == null)
            extractor = extractorViewComponent.createObject(page)
    }

    onStatusChanged: {
        if (status == PageStatus.Activating) {
            TrackerEngine.setSortOptions(sortBy, sortOrder)
            TrackerEngine.currentPath = page.path.length > 0 ? page.path : ""
            extractor.reset()
            directoryPathModel.path = page.path.length > 0 ? page.path : ""
            pathBar.updatePosition()
        }
    }

    Component {
        id: extractorViewComponent

        ExtractorView {
            objectName: "ExtractorView"
            rejectedPageName: "__directory_page"

            onShowInfo: {
                if (!errorNotification) {
                    errorNotification = errorNotificationComponent.createObject(page)
                }
                errorNotification.show(info)
            }
        }
    }

    PopupMenuMain {
        id: headerPopUpMenu

        onOpenedChanged: page.showNavigationIndicator = !opened
        onNewDirectoryClicked: page.openNewDirectoryDialog()
        onDeleteDirectoryClicked:  {
            var _page = pageStack.previousPage(page)
            if (_page && _page.hasOwnProperty("__directory_page")) {
                pageStack.pop()
                _page._deletingPath = page.path
                var popup = Remorse.popupAction(_page,
                                                Remorse.deletedText,
                                                function() {
                                                    _page._commitDeletion()
                                                })
                popup.canceled.connect(function() { _page._deletingPath = "" })
            } else {
                //% "Deleting folder"
                Remorse.popupAction(page, qsTrId("filemanager-la-deleting_folder"), function() {
                    FileEngine.deleteFiles(path, true)
                    folderDeleted(path)
                    pageStack.pop()
                })
            }
        }

        onSelectModeClicked: {
            pageStack.animatorPush(Qt.resolvedUrl("DirectoryPage.qml"),
                                   {
                                       path: page.path,
                                       initialPath: page.initialPath,
                                       extractor: page.extractor,
                                       multiselection: true
                                   },
                                   PageStackAction.Animated)
        }

        onPasteModeClicked: FileEngine.pasteFiles(page.path, true, indexIfExists)
        onSortClicked: sortPopupMenu.sortClicked(sortOrder, sortBy, directorySort)

        pageHeaderMode: true
        isUserDir: FileEngine.isUserDir(page.path)
        showNewFolder: page.showNewFolder
        hasFiles: fileModel.count > 0
        isSelectionMode: gridView.selectionMode
        isSortMode: appBar.isSortingMode
        appBar: appBar
    }

    PopupMenuSort {
        id: sortPopupMenu

        onOpenedChanged: page.showNavigationIndicator = !opened
        onSortClicked: {
            if (sortBy !== fileModel.sortBy || sortOrder !== fileModel.sortOrder || sortBy === FileModel.SortBySize) {
                SortingSettings.setSortType(sortBy, sortOrder, directorySort)
                TrackerEngine.setSortOptions(SortingSettings.sortBy, SortingSettings.sortOrder)
            }
        }
    }

    FileModel {
        id: fileModel

        directorySort: SortingSettings.directorySort
        sortBy: SortingSettings.sortBy
        sortOrder: SortingSettings.sortBy === FileModel.SortBySize
                   ? SortingSettings.sortOrder == Qt.AscendingOrder ? Qt.DescendingOrder : Qt.AscendingOrder
                   : SortingSettings.sortOrder
        path: initialPath
        active: page.status === PageStatus.Active
        includeHiddenFiles: FileManager.hiddenItemsSetting.value
    }

    FileGridHeader {
        id: appBar

        headerTitle: fileModel.path === StandardPaths.home ? qsTrId("files-la-device") : fileModel.path.split("/").pop()
        parentGridView: gridView
        errorType: fileModel.errorType
        path: fileModel.path
        countFiles: fileModel.count
        popUpMenu: headerPopUpMenu
        popUpMenuSort: sortPopupMenu
        isSelectionMode: gridView.selectionMode
        isPasteMode: gridView.pasteMode && !isArchiveMode
        isArchiveMode: gridView.archiveMode
        isExtractMode: gridView.extractMode
        description: page.description
        pageTitle: page.title
        initialPath: page.initialPath
        standardPaths: page.standardPaths
        z: page.z + 1

        onNewDirectoryDialog: page.openNewDirectoryDialog()
        onSearchModeClicked: {
            pageStack.animatorPush(Qt.resolvedUrl("SearchPage.qml"), {
                                       path: page.path,
                                       initialPath: page.initialPath,
                                       caseSensitivity: page.caseSensitivity,
                                       extractor: extractor
                                   })
        }
        onUpdateButtonVisibilityFinished: FileManager.updatePopUpContent()
    }

    PathBar {
        id: pathBar

        anchors {
            top: appBar.bottom
            topMargin: Theme.paddingMedium
        }

        z: appBar.z
        model: directoryPathModel
        enabled: errorType !== FileModel.ErrorNotExist
        visible: enabled
    }

    FileGridView {
        id: gridView
        objectName: "FileGridView"

        anchors.fill: parent
        anchors.topMargin: pathBar.height + appBar.height
        clip: true

        gridLayout: FileManager.gridLayoutSetting.value
        cellWidthForSmallScreen: gridView.width
        model: fileModel
        extractor: page.extractor
        initialPath: page.initialPath
        opacity: {
            if (FileEngine.busy) {
                return 0.6
            } else if (page.mounting) {
                return 0.0
            } else {
                return 1.0
            }
        }
        standardPaths: page.standardPaths
        selectionMode: page.multiselection

        onMultiselectRequest: headerPopUpMenu.selectModeClicked()

        Behavior on opacity { FadeAnimator {} }

        DirectoryPathModel {
            id: directoryPathModel
        }

        header: Column {
            width: parent.width

            SectionHeader {
                objectName: "SortedBySectionHeader"

                x: Theme.paddingLarge
                horizontalAlignment: Text.AlignLeft
                topPadding: -Theme.paddingLarge
                //% "Sorted by %1"
                text: qsTrId("filemanager-la-sort").arg(SortingSettings.selectedSortType)
                opacity: fileModel.count > 0 ? 1 : 0

                Behavior on opacity { FadeAnimation {} }
            }
            // Padding to match design
            Item {
                width: parent.width
                height: Theme.paddingSmall
                opacity: gridView.gridLayout ? 1 : 0

                Behavior on opacity { FadeAnimation {} }
            }
        }

        delegate: FileGridItem {
            id: delegateItem

            function openDirectory(dirPath) {
                FileManager.openDirectory({
                                              path: dirPath,
                                              initialPath: page.initialPath,
                                              caseSensitivity: page.caseSensitivity,
                                              extractor: extractor
                                          })
            }

            enabled: {
                if (gridView.isRemoving)
                    return false

                if (!gridView.pasteMode && !gridView.extractMode && !gridView.isSelectOneItem) {
                    return true
                } else {
                    return model.isDir
                }
            }
            opacity: enabled ? 1.0 : Theme.opacityLow
            fileName: model.absolutePath
            down: model.isSelected || FileEngine.clipboardFiles.indexOf(model.absolutePath) !== -1 || pressed
                  || (isArchive && extractor.archiveFile === fileName)

            onClicked: {
                if (gridView.selectionMode) {
                    fileModel.toggleSelectedFile(model.index)
                } else if (model.isDir) {
                    var dirPath = fileModel.appendPath(model.fileName)
                    if (!FileEngine.isSymLink(dirPath)) {
                        openDirectory(dirPath)
                    } else {
                        if (openSymLink) {
                            openDirectory(dirPath)
                        } else {
                            //% "Following a symbolic link is prohibited"
                            errorNotification.show(qsTrId("filemanager-la-open_symlink_failed"))
                        }
                    }
                } else {
                    if (!FileEngine.isArchive(model.absolutePath)) {
                        if (!FileManager.openGalleryPreviewPage(model.absolutePath, model.mimeType, extractor))
                            Qt.openUrlExternally(FileManager.pathToUrl(model.absolutePath))
                    } else {
                        FileManager.openArchive(model.absolutePath, "/", fileModel.path, extractor, true, archivePageComponent, "__directory_page")
                    }
                }
            }

            onPressAndHold: {
                if (!gridView.selectionMode && !gridView.pasteMode && !gridView.extractMode && !gridView.isSelectOneItem) {
                    gridView.currentIndex = model.index
                    gridView.isSelectOneItem = true
                    fileModel.toggleSelectedFile(model.index)
                    gridView.openPopupMenu(null, delegateItem)
                }
            }

            Component {
                id: archivePageComponent

                ArchivePage {}
            }
        }

        VerticalScrollDecorator {}    

        NoFilesPagePlaceholder {
            visible: (fileModel.count === 0 && fileModel.populated) ||
                     errorType !== FileModel.NoError

            label.text: errorType === FileModel.NoError
                          //% "No files"
                        ? qsTrId("filemanager-la-no_files")
                          //% "No permissions to access %1"
                        : qsTrId("filemanager-la-folder_no_permission_to_access").arg(
                              fileModel.path)
        }
    }

    Component {
        id: errorNotificationComponent

        FileManagerNotification {
            property var connections: Connections {
                target: FileEngine
                onError: {
                    switch (error) {
                    case FileEngine.ErrorOperationInProgress:
                        //% "File operation already in progress"
                        show(qsTrId("filemanager-la-operation_already_in_progress"))
                        break
                    case FileEngine.ErrorCopyFailed:
                        //% "Copying failed"
                        show(qsTrId("filemanager-la-copy_failed"))
                        break
                    case FileEngine.ErrorDeleteFailed:
                        //% "Deletion failed"
                        show(qsTrId("filemanager-la-deletion_failed"))
                        break
                    case FileEngine.ErrorMoveFailed:
                        //% "Moving failed"
                        show(qsTrId("filemanager-la-moving_failed"))
                        break
                    case FileEngine.ErrorRenameFailed:
                        //% "Renaming failed"
                        show(qsTrId("filemanager-la-renaming_failed"))
                        break
                    case FileEngine.ErrorCannotCopyIntoItself:
                        //: Shown both when trying to copy file over itself and copying a folder into itself
                        //% "You cannot copy into itself"
                        show(qsTrId("filemanager-la-cannot_copy_into_itself"))
                        break
                    case FileEngine.ErrorCannotMoveIntoItself:
                        //: Shown both when trying to move file over itself and moving a folder into itself
                        //% "You cannot move into itself"
                        show(qsTrId("filemanager-la-cannot_move_into_itself"))
                        break
                    case FileEngine.ErrorFolderCopyFailed:
                        //% "Copying folder failed"
                        show(qsTrId("filemanager-la-copying_folder_failed"))
                        break
                    case FileEngine.ErrorFolderCreationFailed:
                        //% "Could not create folder"
                        show(qsTrId("filemanager-la-folder_creation_failed"))
                        break
                    case FileEngine.ErrorChmodFailed:
                        //% "Could not set permission"
                        show(qsTrId("filemanager-la-set_permissions_failed"))
                        break
                    case FileEngine.ErrorOutOfSpace:
                        //% "There is not enough space on the disk."
                        show(qsTrId("filemanager-la-out_of_space"),
                             //% "Clear memory and try again."
                             qsTrId("filemanager-la-clear_memory"))
                        break
                    }
                    _deletingPath = "" // reset state if deletion failed
                }
                onModeChanged: {
                    if (FileEngine.mode === FileEngine.IdleMode) {
                        _deletingPath = ""
                    }
                }
            }

            property var busyView: BusyView {
                busy: FileEngine.busy
                enabled: busy || page.mounting
                text: {
                    if (page.mounting) {
                        //% "Mounting SD card"
                        return qsTrId("filemanager-la-mounting")
                    } else switch (FileEngine.mode) {
                        case FileEngine.DeleteMode:
                            //% "Deleting"
                            return qsTrId("filemanager-la-deleting")
                        case FileEngine.CopyMode:
                            //% "Copying"
                            return qsTrId("filemanager-la-copying")
                        default:
                            return ""
                    }
                }
            }
        }
    }
}
