/****************************************************************************
**
** SPDX-FileCopyrightText: Copyright 2021-2024 Open Mobile Platform LLC <community@omp.ru>
** SPDX-License-Identifier: Proprietary
**
** Copyright (C) 2016 - 2021 Jolla Ltd.
**
****************************************************************************/

import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.Silica.private 1.0 as Private
import Nemo.FileManager 1.0
import Sailfish.Pickers 1.0
import Sailfish.FileManager 1.0 as FileManager
import Aurora.Controls 1.0

PickerPage {
    id: page

    property alias path: fileModel.path
    property alias includeDirectories: fileModel.includeDirectories
    property alias includeHiddenFiles: fileModel.includeHiddenFiles
    property alias includeSystemFiles: fileModel.includeSystemFiles
    property alias sortBy: fileModel.sortBy
    property alias sortOrder: fileModel.sortOrder
    property alias directorySort: fileModel.directorySort
    property alias nameFilters: fileModel.nameFilters
    property alias caseSensitivity: fileModel.caseSensitivity
    property alias showParentDirectory: gridView.showParentDirectory
    property alias gridLayout: gridView.gridLayout

    signal filePicked(var path)

    function contentTypeForFile(file) {
        if (file.isVideo) {
            return ContentType.Video
        } else if (file.isPicture) {
            return ContentType.Image
        } else if (!file.isDir) {
            var imageName = Theme.iconForMimeType(file.mimeType)

            if (imageName.indexOf("audio") !== -1) {
                return ContentType.Audio
            } else if (imageName.indexOf("note") !== -1
                       || imageName.indexOf("pdf") !== -1
                       || imageName.indexOf("spreadsheet") !== -1
                       || imageName.indexOf("presentation") !== -1
                       || imageName.indexOf("formatted") !== -1
                       || imageName.indexOf("document") !== -1) {
                return ContentType.Document
            }
        }

        return ContentType.File
    }

    title: fileModel.directoryName

    orientationTransitions: Private.PageOrientationTransition {
        fadeTarget: _background ? gridView : __silica_applicationwindow_instance.contentItem
        targetPage: page
    }

    AppBar {
        id: appBar

        headerText: title

        AppBarSpacer {}

        AppBarButton {
            onClicked: sortPopupMenu.open()

            icon.source: "image://theme/icon-splus-sort-by"
            //% "Sort"
            context: qsTrId("components_pickers-la-sort")

            PopupMenu {
                id: sortPopupMenu

                //% "Sort by"
                headerText: qsTrId("components_pickers-he-sort-by")
                headerBackground: PopupMenuHeader.SolidBackground

                SortPopupMenuItem {
                    //% "Name"
                    text: qsTrId("components_pickers-la-sort_name")
                    sortBy: FileModel.SortByName
                    fileModel: fileModel
                }

                SortPopupMenuItem {
                    //% "Size"
                    text: qsTrId("components_pickers-la-sort_size")
                    sortBy: FileModel.SortBySize
                    fileModel: fileModel
                }

                SortPopupMenuItem {
                    //% "Date modified"
                    text: qsTrId("components_pickers-la-sort_date")
                    sortBy: FileModel.SortByModified
                    fileModel: fileModel
                }

                SortPopupMenuItem {
                    //% "Extension"
                    text: qsTrId("components_pickers-la-sort_extension")
                    sortBy: FileModel.SortByExtension
                    fileModel: fileModel
                }
            }
        }

        AppBarButton {
            onClicked: morePopupMenu.open()

            icon.source: "image://theme/icon-splus-more"
            //% "More actions"
            context: qsTrId("components_pickers-la-more-actions")

            PopupMenu {
                id: morePopupMenu

                PopupMenuItem {
                    onClicked: gridLayout = !gridLayout

                    text: gridLayout
                            //% "Show list"
                          ? qsTrId("components_pickers-la-show_grid")
                            //% "Show grid"
                          : qsTrId("components_pickers-la-show_list")
                    icon.source: "image://theme/icon-splus-%1".arg(gridLayout ? "list" : "cage")
                }

                PopupMenuItem {
                    onClicked: includeHiddenFiles= !includeHiddenFiles

                    text: includeHiddenFiles
                            //% "Hide hidden files"
                          ? qsTrId("components_pickers-la-hide_hidden_files")
                            //% "Show hidden files"
                          : qsTrId("components_pickers-la-show_hidden_files")
                    icon.source: "image://theme/icon-splus-file-%1".arg(includeHiddenFiles ? "hide" : "show")
                }
            }
        }
    }

    FileGridView {
        id: gridView

        property bool showParentDirectory

        anchors.fill: parent
        anchors.topMargin: appBar.height + Theme.paddingMedium + SafeZoneRect.insets.top
        cellWidthForSmallScreen: parent.width
        currentIndex: -1
        selectionEnabled: false

        model: FileModel {
            id: fileModel
            active: page.status === PageStatus.Active
            path: StandardPaths.home
            includeDirectories: true
            sortBy: FileModel.SortByName
            directorySort: FileModel.SortDirectoriesBeforeFiles

            includeParentDirectory: gridView.showParentDirectory
        }

        delegate: FileManager.FileGridItem {
            onClicked: {
                if (model.isDir) {
                    var obj = pageStack.animatorPush('DirectoryPage.qml', {
                                                         path: model.absolutePath,
                                                         includeHiddenFiles: fileModel.includeHiddenFiles,
                                                         sortBy: fileModel.sortBy,
                                                         sortOrder: fileModel.sortOrder,
                                                         directorySort: fileModel.directorySort,
                                                         nameFilters: fileModel.nameFilters,
                                                         caseSensitivity: fileModel.caseSensitivity,
                                                         gridLayout: gridLayout,
                                                         allowedOrientations: page.allowedOrientations
                                                     })
                    obj.pageCompleted.connect(function(nextPage) {
                        nextPage.filePicked.connect(page.filePicked)
                    })
                } else {
                    page.filePicked({
                                        'url': model.url,
                                        'title': model.fileName,
                                        'lastModified': model.modified,
                                        'filePath': model.absolutePath,
                                        'fileName': model.fileName,
                                        'fileSize': model.size,
                                        'mimeType': model.mimeType,
                                        'selected': true,
                                        'contentType': page.contentTypeForFile(this)
                                    })
                }
            }

            onPressAndHold: if (!model.isDir) popupMenu.open()

            fileName: model.absolutePath
            size: model.size
            modified: model.modified

            PreviewPopupMenu {
                id: popupMenu

                fileName: model.url
            }
        }

        ViewPlaceholder {
            text: {
                if (page.nameFilters.length) {
                    //: Empty state text if no files match the filter
                    //% "No files match filter"
                    return qsTrId("components_pickers-ph-no_matching_files")
                }
                //: Empty state text if the directory contains no content
                //% "Empty folder"
                return qsTrId("components_pickers-ph-empty_folder")
            }
            enabled: fileModel.populated && fileModel.count === 0 && !page.showParentDirectory
        }

        VerticalScrollDecorator {}
    }
}
